// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.formatting;

import com.intellij.formatting.engine.ExpandableIndent;
import org.jetbrains.annotations.NotNull;

public interface Formatter extends IndentFactory {
  static Formatter getInstance() {
    return Holder.INSTANCE;
  }
}

final class Holder implements Formatter {

  private final IndentImpl NONE_INDENT = new IndentImpl(Indent.Type.NONE, false, false);
  private final IndentImpl myAbsoluteNoneIndent = new IndentImpl(Indent.Type.NONE, true, false);
  private final IndentImpl myLabelIndent = new IndentImpl(Indent.Type.LABEL, false, false);
  private final IndentImpl myContinuationIndentRelativeToDirectParent = new IndentImpl(Indent.Type.CONTINUATION, false, true);
  private final IndentImpl myContinuationIndentNotRelativeToDirectParent = new IndentImpl(Indent.Type.CONTINUATION, false, false);
  private final IndentImpl myContinuationWithoutFirstIndentRelativeToDirectParent
    = new IndentImpl(Indent.Type.CONTINUATION_WITHOUT_FIRST, false, true);
  private final IndentImpl myContinuationWithoutFirstIndentNotRelativeToDirectParent
    = new IndentImpl(Indent.Type.CONTINUATION_WITHOUT_FIRST, false, false);
  private final IndentImpl myAbsoluteLabelIndent = new IndentImpl(Indent.Type.LABEL, true, false);
  private final IndentImpl myNormalIndentRelativeToDirectParent = new IndentImpl(Indent.Type.NORMAL, false, true);
  private final IndentImpl myNormalIndentNotRelativeToDirectParent = new IndentImpl(Indent.Type.NORMAL, false, false);

  // NotNullLazyValue is not used here because ServiceManager.getService can return null and better to avoid any possible issues here
  volatile static Formatter INSTANCE = new Holder();

  @Override
  public Indent getNormalIndent(boolean relative) {
    return relative ? myNormalIndentRelativeToDirectParent : myNormalIndentNotRelativeToDirectParent;
  }

  @Override
  public Indent getNoneIndent() {
    return NONE_INDENT;
  }

  @Override
  public Indent getSpaceIndent(final int spaces, final boolean relative) {
    return getIndent(Indent.Type.SPACES, spaces, relative, false);
  }

  @Override
  public Indent getIndent(@NotNull Indent.Type type, boolean relativeToDirectParent, boolean enforceIndentToChildren) {
    return getIndent(type, 0, relativeToDirectParent, enforceIndentToChildren);
  }

  @Override
  public Indent getSmartIndent(@NotNull Indent.Type type) {
    return new ExpandableIndent(type);
  }

  @Override
  public Indent getIndent(@NotNull Indent.Type type, int spaces, boolean relativeToDirectParent, boolean enforceIndentToChildren) {
    return new IndentImpl(type, false, spaces, relativeToDirectParent, enforceIndentToChildren);
  }

  @Override
  public Indent getAbsoluteLabelIndent() {
    return myAbsoluteLabelIndent;
  }

  @Override
  public Indent getAbsoluteNoneIndent() {
    return myAbsoluteNoneIndent;
  }

  @Override
  public Indent getLabelIndent() {
    return myLabelIndent;
  }

  @Override
  public Indent getContinuationIndent(boolean relative) {
    return relative ? myContinuationIndentRelativeToDirectParent : myContinuationIndentNotRelativeToDirectParent;
  }

  //is default
  @Override
  public Indent getContinuationWithoutFirstIndent(boolean relative) {
    return relative ? myContinuationWithoutFirstIndentRelativeToDirectParent : myContinuationWithoutFirstIndentNotRelativeToDirectParent;
  }
}