// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.psi.templateLanguages;

import com.intellij.lang.ASTNode;
import com.intellij.lang.Language;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.TextRange;
import com.intellij.psi.ArrayTokenSequence;
import com.intellij.psi.TokenType;
import com.intellij.psi.templateLanguages.TemplateDataElementType.OuterLanguageRangePatcher;
import com.intellij.psi.tree.IElementType;
import com.intellij.util.containers.ContainerUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.function.Function;

public class RangeCollectorImpl extends TemplateDataElementType.RangeCollector {
  private static final DefaultOuterLanguagePatcher DEFAULT_OUTER_LANGUAGE_PATCHER = new DefaultOuterLanguagePatcher();

  private final TemplateDataElementType myTemplateDataElementType;
  private final List<TextRange> myOuterAndRemoveRanges;

  static final Key<RangeCollectorImpl> OUTER_ELEMENT_RANGES = Key.create("template.parser.outer.element.handler");

  RangeCollectorImpl(@NotNull TemplateDataElementType templateDataElementType) {
    this(templateDataElementType, new ArrayList<>());
  }

  private RangeCollectorImpl(@NotNull TemplateDataElementType templateDataElementType, @NotNull List<TextRange> outerAndRemoveRanges) {
    myTemplateDataElementType = templateDataElementType;
    myOuterAndRemoveRanges = outerAndRemoveRanges;
  }

  @Override
  public void addOuterRange(@NotNull TextRange newRange) {
    addOuterRange(newRange, false);
  }

  @Override
  public void addOuterRange(@NotNull TextRange newRange, boolean isInsertion) {
    if (newRange.isEmpty()) {
      return;
    }
    assertRangeOrder(newRange);

    if (!myOuterAndRemoveRanges.isEmpty()) {
      int lastItemIndex = myOuterAndRemoveRanges.size() - 1;
      TextRange lastRange = myOuterAndRemoveRanges.get(lastItemIndex);
      if (lastRange.getEndOffset() == newRange.getStartOffset() && !(lastRange instanceof RangeToRemove)) {
        TextRange joinedRange =
          lastRange instanceof InsertionRange || isInsertion
          ? new InsertionRange(lastRange.getStartOffset(), newRange.getEndOffset())
          : TextRange.create(lastRange.getStartOffset(), newRange.getEndOffset());
        myOuterAndRemoveRanges.set(lastItemIndex, joinedRange);
        return;
      }
    }
    myOuterAndRemoveRanges.add(isInsertion ? new InsertionRange(newRange.getStartOffset(), newRange.getEndOffset()) : newRange);
  }

  @Override
  public void addRangeToRemove(@NotNull TextRange rangeToRemove) {
    if (rangeToRemove.isEmpty()) {
      return;
    }
    assertRangeOrder(rangeToRemove);

    myOuterAndRemoveRanges.add(new RangeToRemove(rangeToRemove.getStartOffset(), rangeToRemove.getEndOffset()));
  }

  private void assertRangeOrder(@NotNull TextRange newRange) {
    TextRange range = ContainerUtil.getLastItem(myOuterAndRemoveRanges);
    assert range == null || newRange.getStartOffset() >= range.getStartOffset();
  }

  void prepareFileForParsing(@NotNull Language templateLanguage,
                             @NotNull CharSequence originalSourceCode,
                             @NotNull CharSequence templateSourceCode) {
    addDummyStringsToRangesToRemove(templateSourceCode);
    OuterLanguageRangePatcher patcher = DEFAULT_OUTER_LANGUAGE_PATCHER;
    StringBuilder builder =
      templateSourceCode instanceof StringBuilder ? (StringBuilder)templateSourceCode : new StringBuilder(templateSourceCode);
    insertDummyStringIntoInsertionRanges(patcher, originalSourceCode, builder);
  }

  /**
   * Sets {@link RangeToRemove#myTextToRemove} in {@link #myOuterAndRemoveRanges} from generated file,
   * so lazy parseables implementing {@link TemplateDataElementType.TemplateAwareElementType} will have dummy strings when they are parsed.
   */
  private void addDummyStringsToRangesToRemove(@NotNull CharSequence generatedText) {
    int shift = 0;
    ListIterator<TextRange> iterator = myOuterAndRemoveRanges.listIterator();
    while (iterator.hasNext()) {
      TextRange range = iterator.next();
      if (range instanceof RangeToRemove) {
        CharSequence insertedString = generatedText.subSequence(range.getStartOffset() - shift, range.getEndOffset() - shift);
        iterator.set(new RangeToRemove(range.getStartOffset(), insertedString)); // only add insertedString
        shift -= range.getLength();
      }
      else {
        shift += range.getLength();
      }
    }
  }

  private void insertDummyStringIntoInsertionRanges(@NotNull OuterLanguageRangePatcher patcher,
                                                    @NotNull CharSequence originalSourceCode,
                                                    @NotNull StringBuilder modifiedText) {
    if (myOuterAndRemoveRanges.isEmpty()) return;

    int shift = 0;
    ListIterator<TextRange> iterator = myOuterAndRemoveRanges.listIterator();
    while (iterator.hasNext()) {
      TextRange outerElementRange = iterator.next();
      if (outerElementRange instanceof RangeToRemove) {
        shift += outerElementRange.getLength();
      }
      else {
        if (outerElementRange instanceof InsertionRange) {
          String dummyString = patcher.getTextForOuterLanguageInsertionRange(
            myTemplateDataElementType,
            outerElementRange.subSequence(originalSourceCode));
          if (dummyString != null) {
            // Don't set RangeToRemove#myTextToRemove so it won't be applied to nested lazy parseables.
            // Nested lazy parseables may add dummy string themselves.
            iterator.add(new RangeToRemove(outerElementRange.getEndOffset(), outerElementRange.getEndOffset() + dummyString.length()));
            modifiedText.insert(outerElementRange.getStartOffset() + shift, dummyString);
            shift += dummyString.length();
          }
        }
        shift -= outerElementRange.getLength();
      }
    }
  }

  ArrayTokenSequence insertOuterElementsAndRemoveRanges(@NotNull ArrayTokenSequence modifiedDataTokens) {
    int lexemeCount = modifiedDataTokens.getLexemeCount();
    int[] lexStarts = new int[lexemeCount + 2];
    IElementType[] lexTypes = new IElementType[lexemeCount + 2];
    int shift = 0;
    Iterator<TextRange> rangeToProcessIterator = myOuterAndRemoveRanges.iterator();
    TextRange rangeToProcess = rangeToProcessIterator.hasNext() ? rangeToProcessIterator.next() : null;
    for (int i = 0; i < lexemeCount + 1; i++) {
      while (rangeToProcess != null && rangeToProcess.getStartOffset() - shift < modifiedDataTokens.lexStart(i)) {
        if (rangeToProcess instanceof RangeToRemove) {
          shift -= rangeToProcess.getLength();
        }
        else {
          shift += rangeToProcess.getLength();
        }
        rangeToProcess = rangeToProcessIterator.hasNext() ? rangeToProcessIterator.next() : null;
      }
      lexStarts[i] = modifiedDataTokens.lexStart(i) + shift;
      lexTypes[i] = modifiedDataTokens.lexType(i);
    }
    if (rangeToProcess != null) {
      lexStarts[lexemeCount + 1] = rangeToProcess.getEndOffset();
      lexTypes[lexemeCount] = TokenType.WHITE_SPACE;
      lexemeCount++;
    }

    return new ArrayTokenSequence(lexStarts, lexTypes, lexemeCount);
  }


  /**
   * Like {@link TemplateDataElementType#parse} builds the tree considering outer language elements, but for inner lazy parseables.
   */
  @NotNull ASTNode applyRangeCollectorAndExpandChameleon(@NotNull ASTNode chameleon,
                                                         @NotNull Language language,
                                                         @NotNull Function<? super @NotNull CharSequence, ? extends @NotNull ASTNode> parser) {
    CharSequence chars = chameleon.getChars();
    if (myOuterAndRemoveRanges.isEmpty()) return parser.apply(chars);

    StringBuilder stringBuilder = applyOuterAndRemoveRanges(chars);

    ASTNode root = parser.apply(stringBuilder.toString());

    return root;
  }

  @NotNull
  private StringBuilder applyOuterAndRemoveRanges(CharSequence chars) {
    StringBuilder stringBuilder = new StringBuilder(chars);
    int shift = 0;
    for (TextRange outerElementRange : myOuterAndRemoveRanges) {
      if (outerElementRange instanceof RangeToRemove) {
        CharSequence textToRemove = ((RangeToRemove)outerElementRange).myTextToRemove;
        if (textToRemove != null) {
          stringBuilder.insert(outerElementRange.getStartOffset() + shift, textToRemove);
          shift += textToRemove.length();
        }
      }
      else {
        stringBuilder.delete(outerElementRange.getStartOffset() + shift,
                             outerElementRange.getEndOffset() + shift);
        shift -= outerElementRange.getLength();
      }
    }
    return stringBuilder;
  }

  @NotNull CharSequence applyTemplateDataModifications(@NotNull CharSequence sourceCode, @NotNull TemplateDataModifications modifications) {
    assert myOuterAndRemoveRanges.isEmpty();
    List<TextRange> ranges = modifications.myOuterAndRemoveRanges;
    if (ranges.isEmpty()) return sourceCode;
    for (TextRange range : ranges) {
      if (range instanceof RangeToRemove) {
        if (range.isEmpty()) continue;
        assertRangeOrder(range);
        CharSequence textToRemove = ((RangeToRemove)range).myTextToRemove;
        assert textToRemove != null;
        myOuterAndRemoveRanges.add(new RangeToRemove(range.getStartOffset(), textToRemove));
      }
      else {
        addOuterRange(range, range instanceof InsertionRange);
      }
    }

    return applyOuterAndRemoveRanges(sourceCode);
  }


  final static class RangeToRemove extends TextRange {
    /**
     * We need this text to propagate dummy strings through lazy parseables. If this text is null, dummy identifier won't be propagated.
     */
    public final @Nullable CharSequence myTextToRemove;

    RangeToRemove(int startOffset, @NotNull CharSequence text) {
      super(startOffset, startOffset + text.length());
      myTextToRemove = text;
    }

    RangeToRemove(int startOffset, int endOffset) {
      super(startOffset, endOffset);
      myTextToRemove = null;
    }

    @Override
    public @NotNull TextRange shiftLeft(int delta) {
      if (delta == 0) return this;
      return myTextToRemove != null
             ? new RangeToRemove(getStartOffset() - delta, myTextToRemove)
             : new RangeToRemove(getStartOffset() - delta, getEndOffset() - delta);
    }

    @Override
    public @NotNull TextRange intersection(@NotNull TextRange range) {
      int newStart = Math.max(getStartOffset(), range.getStartOffset());
      int newEnd = Math.min(getEndOffset(), range.getEndOffset());
      assertProperRange(newStart, newEnd, "Invalid range");
      return myTextToRemove != null
             ? new RangeToRemove(newStart, myTextToRemove.subSequence(newStart - getStartOffset(), newEnd - getStartOffset()))
             : new RangeToRemove(newStart, newEnd);
    }

    @Override
    public String toString() {
      return "RangeToRemove" + super.toString();
    }
  }

  static final class InsertionRange extends TextRange {

    InsertionRange(int startOffset, int endOffset) {
      super(startOffset, endOffset);
    }

    @Override
    public @NotNull TextRange shiftLeft(int delta) {
      if (delta == 0) return this;
      return new InsertionRange(getStartOffset() - delta, getEndOffset() - delta);
    }

    @Override
    public @NotNull TextRange intersection(@NotNull TextRange range) {
      int newStart = Math.max(getStartOffset(), range.getStartOffset());
      int newEnd = Math.min(getEndOffset(), range.getEndOffset());
      assertProperRange(newStart, newEnd, "Invalid range");
      return new InsertionRange(newStart, newEnd);
    }

    @Override
    public String toString() {
      return "InsertionRange" + super.toString();
    }
  }
}
