/*
 * Copyright 2000-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.intellij.psi.builder;

import com.intellij.AyaModified;
import com.intellij.openapi.diagnostic.Logger;
import kala.collection.base.primitive.IntIterator;
import kala.collection.mutable.primitive.MutableIntArrayList;
import kala.collection.mutable.primitive.MutableIntList;
import kala.function.IntHasher;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.Range;

import java.util.Arrays;

/**
 * @author peter
 */
@AyaModified
final class MarkerProduction {
  private final MutableIntList delegate = MutableIntArrayList.create(256);
  private static final Logger LOG = Logger.getInstance(MarkerProduction.class);
  private static final int LINEAR_SEARCH_LIMIT = 20;
  private final MarkerPool myPool;
  private final MarkerOptionalData myOptionalData;

  MarkerProduction(MarkerPool pool, MarkerOptionalData optionalData) {
    myPool = pool;
    myOptionalData = optionalData;
  }

  void addBefore(FleetPsiBuilder.ProductionMarker marker, FleetPsiBuilder.ProductionMarker anchor) {
    insert(indexOf(anchor), marker.markerId);
  }

  @AyaModified public int getInt(int index) {
    return get(index);
  }

  @SuppressWarnings("UnusedReturnValue")
  @AyaModified public int removeInt(int index) {
    return removeAt(index);
  }

  private int indexOf(FleetPsiBuilder.ProductionMarker marker) {
    int idx = findLinearly(marker.markerId);
    if (idx < 0) {
      for (int i = findMarkerAtLexeme(marker.getLexemeIndex(false)); i < size(); i++) {
        if (getInt(i) == marker.markerId) {
          return i;
        }
      }
    }
    if (idx < 0) {
      LOG.error("Dropped or rolled-back marker");
    }
    return idx;
  }

  private int findLinearly(int markerId) {
    int low = Math.max(0, size() - LINEAR_SEARCH_LIMIT);
    for (int i = size() - 1; i >= low; i--) {
      if (getInt(i) == markerId) {
        return i;
      }
    }
    return -1;
  }

  private int findMarkerAtLexeme(int lexemeIndex) {
    int i = binarySearch(0, size() - LINEAR_SEARCH_LIMIT, mid -> Integer.compare(getLexemeIndexAt(mid), lexemeIndex));
    return i < 0 ? -1 : findSameLexemeGroupStart(lexemeIndex, i);
  }

  public int size() {
    return delegate.size();
  }

  public boolean isEmpty() {
    return delegate.isEmpty();
  }

  public boolean isNotEmpty() {
    return delegate.isNotEmpty();
  }

  public int get(@Range(from = 0L, to = 2147483647L) int index) {
    return delegate.get(index);
  }

  public void append(int value) {
    delegate.append(value);
  }

  public void prepend(int value) {
    delegate.prepend(value);
  }

  public void insert(int index, int value) {
    delegate.insert(index, value);
  }

  public int removeAt(int index) {
    return delegate.removeAt(index);
  }

  public void clear() {
    delegate.clear();
  }

  public void set(int index, int newValue) {
    delegate.set(index, newValue);
  }

  public @NotNull IntIterator iterator() {
    return delegate.iterator();
  }

  public int lastIndexOf(int value) {
    for (int i = delegate.size() - 1; i > 0; i--)
      if (delegate.get(i) == value) return i;
    return -1;
  }

  private static int binarySearch(int fromIndex, int toIndex, @NotNull IntHasher indexComparator) {
    int low = fromIndex;
    int high = toIndex - 1;
    while (low <= high) {
      int mid = (low + high) >>> 1;
      int cmp = indexComparator.hash(mid);
      if (cmp < 0) low = mid + 1;
      else if (cmp > 0) high = mid - 1;
      else return mid;
    }
    return -(low + 1);
  }


  private int findSameLexemeGroupStart(int lexemeIndex, int prodIndex) {
    while (prodIndex > 0 && getLexemeIndexAt(prodIndex - 1) == lexemeIndex) prodIndex--;
    return prodIndex;
  }

  void addMarker(FleetPsiBuilder.ProductionMarker marker) {
    append(marker.markerId);
  }

  void rollbackTo(FleetPsiBuilder.ProductionMarker marker) {
    int idx = indexOf(marker);
    for (int i = size() - 1; i >= idx; i--) {
      int markerId = getInt(i);
      if (markerId > 0) {
        myPool.freeMarker(myPool.get(markerId));
      }
      removeInt(i);
    }
  }

  boolean hasErrorsAfter(@NotNull FleetPsiBuilder.StartMarker marker) {
    for (int i = indexOf(marker) + 1; i < size(); ++i) {
      FleetPsiBuilder.ProductionMarker m = getStartMarkerAt(i);
      if (m != null && hasError(m)) return true;
    }
    return false;
  }

  private boolean hasError(FleetPsiBuilder.ProductionMarker marker) {
    return marker instanceof FleetPsiBuilder.ErrorItem || myOptionalData.getDoneError(marker.markerId) != null;
  }

  void dropMarker(@NotNull FleetPsiBuilder.StartMarker marker) {
    if (marker.isDone()) {
      removeInt(lastIndexOf(-marker.markerId));
    }
    removeInt(indexOf(marker));
    myPool.freeMarker(marker);
  }

  void addDone(FleetPsiBuilder.StartMarker marker, @Nullable FleetPsiBuilder.ProductionMarker anchorBefore) {
    insert(anchorBefore == null ? size() : indexOf(anchorBefore), -marker.markerId);
  }

  @Nullable
  FleetPsiBuilder.ProductionMarker getMarkerAt(int index) {
    int id = getInt(index);
    return myPool.get(id > 0 ? id : -id);
  }

  @Nullable
  FleetPsiBuilder.ProductionMarker getStartMarkerAt(int index) {
    int id = getInt(index);
    return id > 0 ? myPool.get(id) : null;
  }

  @Nullable
  FleetPsiBuilder.StartMarker getDoneMarkerAt(int index) {
    int id = getInt(index);
    return id < 0 ? (FleetPsiBuilder.StartMarker) myPool.get(-id) : null;
  }

  int getLexemeIndexAt(int productionIndex) {
    int id = getInt(productionIndex);
    return myPool.get(Math.abs(id)).getLexemeIndex(id < 0);
  }

  void confineMarkersToMaxLexeme(int markersBefore, int lexemeIndex) {
    for (int k = markersBefore - 1; k > 1; k--) {
      int id = getInt(k);
      FleetPsiBuilder.ProductionMarker marker = myPool.get(Math.abs(id));
      boolean done = id < 0;
      if (marker.getLexemeIndex(done) < lexemeIndex) break;

      marker.setLexemeIndex(lexemeIndex, done);
    }
  }

  @SuppressWarnings("UseOfSystemOutOrSystemErr")
  void doHeavyChecksOnMarkerDone(@NotNull FleetPsiBuilder.StartMarker doneMarker, @Nullable FleetPsiBuilder.StartMarker anchorBefore) {
    int idx = indexOf(doneMarker);

    int endIdx = size();
    if (anchorBefore != null) {
      endIdx = indexOf(anchorBefore);
      if (idx > endIdx) {
        LOG.error("'Before' marker precedes this one.");
      }
    }

    for (int i = endIdx - 1; i > idx; i--) {
      FleetPsiBuilder.ProductionMarker item = getStartMarkerAt(i);
      if (item instanceof FleetPsiBuilder.StartMarker otherMarker) {
        if (!otherMarker.isDone()) {
          Throwable debugAllocThis = myOptionalData.getAllocationTrace(doneMarker);
          Throwable currentTrace = new Throwable();
          if (debugAllocThis != null) {
            makeStackTraceRelative(debugAllocThis, currentTrace).printStackTrace(System.err);
          }
          Throwable debugAllocOther = myOptionalData.getAllocationTrace(otherMarker);
          if (debugAllocOther != null) {
            makeStackTraceRelative(debugAllocOther, currentTrace).printStackTrace(System.err);
          }
          LOG.error("Another not done marker added after this one. Must be done before this.");
        }
      }
    }
  }

  @NotNull
  private static Throwable makeStackTraceRelative(@NotNull Throwable th, @NotNull Throwable relativeTo) {
    StackTraceElement[] trace = th.getStackTrace();
    StackTraceElement[] rootTrace = relativeTo.getStackTrace();
    for (int i = 0, len = Math.min(trace.length, rootTrace.length); i < len; i++) {
      if (trace[trace.length - i - 1].equals(rootTrace[rootTrace.length - i - 1])) continue;
      int newDepth = trace.length - i;
      th.setStackTrace(Arrays.copyOf(trace, newDepth));
      break;
    }
    return th;
  }

  void assertNoDoneMarkerAround(@NotNull FleetPsiBuilder.StartMarker pivot) {
    int pivotIndex = indexOf(pivot);
    for (int i = pivotIndex + 1; i < size(); i++) {
      FleetPsiBuilder.StartMarker m = getDoneMarkerAt(i);
      if (m != null && m.myLexemeIndex <= pivot.myLexemeIndex && indexOf(m) < pivotIndex) {
        throw new AssertionError("There's a marker of type '" + m.getTokenType() + "' that starts before and finishes after the current marker. See cause for its allocation trace.", myOptionalData.getAllocationTrace(m));
      }
    }
  }

}
