package com.intellij.psi.builder;

import com.intellij.AyaModified;
import com.intellij.psi.ArrayTokenSequence;
import com.intellij.psi.FleetPsiParser;
import com.intellij.psi.tree.ILazyParseableElementType;
import com.intellij.psi.tree.TokenSet;
import org.jetbrains.annotations.NotNull;

@AyaModified
public record ASTMarkerVisitor(
  @NotNull FleetPsiParser psiParser,
  @NotNull ASTMarkers<?> root,
  @NotNull TokenSet whitespaces,
  @NotNull TokenSet comments,
  @NotNull ArrayTokenSequence tokens,
  @NotNull String text
) {
  public void visitTree(
    @NotNull ASTMarkers<?> astMarkers,
    int startLexemeOffset,
    int currentMarker,
    int currentOffset,
    @NotNull MarkerNode parent
  ) {
    if (currentMarker >= astMarkers.getSize()) return;
    if (astMarkers.elementType(currentMarker) instanceof ILazyParseableElementType elementType && astMarkers.collapsed(currentMarker)) {
      parseChameleon(
        startLexemeOffset,
        astMarkers.lexemeCount(currentMarker),
        elementType,
        parent,
        currentOffset
      );
      return;
    }

    ASTMarkers.check(!(astMarkers.kind(currentMarker) != ASTMarkers.MarkerKind.Start && astMarkers.kind(currentMarker) != ASTMarkers.MarkerKind.Error), "");
    if (astMarkers.kind(currentMarker) == ASTMarkers.MarkerKind.Error) {
      return;
    }
    var endLexemeOffset = getEndLexemeIndex(astMarkers, startLexemeOffset, currentMarker);
    var child = astMarkers.firstChild(currentMarker);
    var prevOffset = startLexemeOffset;
    var i = startLexemeOffset;
    while (i < endLexemeOffset) {
      if (child != -1 && prevOffset + astMarkers.lexemeRelOffset(child) == i) {
        var childStartLexemeOffset = prevOffset + astMarkers.lexemeRelOffset(child);
        i = getEndLexemeIndex(astMarkers, childStartLexemeOffset, child);
        prevOffset = i;
        var node = new MarkerNode(
          astMarkers.elementType(child),
          offsetByLexemeIndex(childStartLexemeOffset) + currentOffset,
          offsetByLexemeIndex(i) + currentOffset,
          parent);
        visitTree(astMarkers, childStartLexemeOffset, child, currentOffset, node);
        child = astMarkers.nextSibling(child);
      } else {
        var node = new MarkerNode(
          tokens.lexType(i),
          offsetByLexemeIndex(i) + currentOffset,
          offsetByLexemeIndex(i + 1) + currentOffset,
          parent);
        if (tokens.lexType(i) instanceof ILazyParseableElementType lexType &&
          parent.compareAsToken(node)) {
          parseChameleon(i, 1, lexType, node, currentOffset);
          i++;
          continue;
        }
        i++;
      }
    }
    while (child != -1) {
      new MarkerNode(
        astMarkers.elementType(child),
        offsetByLexemeIndex(i) + currentOffset,
        offsetByLexemeIndex(i + astMarkers.lexemeRelOffset(child)) + currentOffset,
        parent);
      i += astMarkers.lexemeRelOffset(child);
      child = astMarkers.nextSibling(child);
    }
  }

  private void parseChameleon(
    int startLexemeOffset,
    int lexCount,
    @NotNull ILazyParseableElementType elementType,
    @NotNull MarkerNode parent,
    int currentOffset
  ) {
    var lexer = elementType.createInnerLexer();
    // note: kotlin code here uses `Any?` as generic type, I am quite confused.
    FleetPsiBuilder<ASTMarkers<Object>> chameleonPsiBuilder;
    ArrayTokenSequence chameleonTokens;
    String chameleonText;
    int lexemeChameleonStart;
    int newCurrentOffset;

    if (lexer != null) {
      var beginIndex = offsetByLexemeIndex(startLexemeOffset);
      var endIndex = offsetByLexemeIndex(startLexemeOffset + lexCount);
      newCurrentOffset = beginIndex + currentOffset;
      chameleonText = text.substring(beginIndex, endIndex);
      chameleonTokens = new ArrayTokenSequence.Builder(chameleonText, lexer).performLexing();
      lexemeChameleonStart = 0;
      chameleonPsiBuilder = psiParser.wrap(new MarkerPsiBuilder<>(
        chameleonText,
        chameleonTokens,
        whitespaces,
        comments,
        0, chameleonTokens.getLexemeCount()));
    } else {
      chameleonTokens = tokens;
      newCurrentOffset = currentOffset;
      lexemeChameleonStart = startLexemeOffset;
      chameleonText = text;
      chameleonPsiBuilder = psiParser.wrap(new MarkerPsiBuilder<>(
        text,
        tokens,
        whitespaces,
        comments,
        startLexemeOffset, lexCount));
    }
    elementType.parse(chameleonPsiBuilder);
    var astMarkers = (ASTMarkers<?>) chameleonPsiBuilder.getRoot();
    if (astMarkers.getSize() != 0) {
      var treePrinter = new ASTMarkerVisitor(
        psiParser,
        chameleonPsiBuilder.getRoot(),
        whitespaces,
        comments,
        chameleonTokens,
        chameleonText);
      treePrinter.visitTree(
        chameleonPsiBuilder.getRoot(),
        lexemeChameleonStart,
        0,
        newCurrentOffset,
        parent);
    }
  }

  private int offsetByLexemeIndex(int i) {
    return i > tokens.getLexemeCount() ? text.length() : tokens.lexStart(i);
  }

  private int getEndLexemeIndex(@NotNull ASTMarkers<?> astMarkers, int startLexemeIndex, int index) {
    return switch (astMarkers.kind(index)) {
      case ASTMarkers.MarkerKind.Error, ASTMarkers.MarkerKind.Start -> startLexemeIndex + astMarkers.lexemeCount(index);
      default -> throw new IllegalStateException("No default");
    };
  }
}
