// Copyright 2000-2020 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.lang;

import com.intellij.openapi.fileTypes.LanguageFileType;
import com.intellij.psi.tree.IElementType;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * The base class for all programming language support implementations.
 * Specific language implementations should inherit from this class
 * and its registered instance wrapped with {@link LanguageFileType} via {@code com.intellij.fileType} extension point.
 * There should be exactly one instance of each Language.
 * It is usually created when creating {@link LanguageFileType} and can be retrieved later with {@link #findInstance(Class)}.
 */
public abstract class Language {
  private static final Map<Class<? extends Language>, Language> ourRegisteredLanguages = new ConcurrentHashMap<>();
  private static final ConcurrentMap<String, List<Language>> ourRegisteredMimeTypes = new ConcurrentHashMap<>();
  private static final Map<String, Language> ourRegisteredIDs = new ConcurrentHashMap<>();

  private final Language myBaseLanguage;
  private final String myID;
  private final String[] myMimeTypes;
  private final List<Language> myDialects = new ArrayList<>();

  public static final Language ANY = new Language("") {
    @Override
    public String toString() {
      return "Language: ANY";
    }
  };

  protected Language(@NonNls @NotNull String ID) {
    this(ID, new String[0]);
  }

  protected Language(@NonNls @NotNull String ID, @NonNls String @NotNull ... mimeTypes) {
    this(null, ID, mimeTypes);
  }

  protected Language(@Nullable Language baseLanguage, @NonNls @NotNull String ID, @NonNls String @NotNull ... mimeTypes) {
    myBaseLanguage = baseLanguage;
    myID = ID;
    myMimeTypes = mimeTypes.length == 0 ? new String[0] : mimeTypes;

    Class<? extends Language> langClass = getClass();
    Language prev = ourRegisteredLanguages.put(langClass, this);
    if (prev != null) {
      throw new IllegalStateException("Language of '" + langClass + "' is already registered: " + prev);
    }

    prev = ourRegisteredIDs.put(ID, this);
    if (prev != null) {
      throw new IllegalStateException("Language with ID '" + ID + "' is already registered: " + prev.getClass());
    }

    for (String mimeType : mimeTypes) {
      if (mimeType == null || mimeType.isEmpty()) {
        continue;
      }
      List<Language> languagesByMimeType = ourRegisteredMimeTypes.get(mimeType);
      if (languagesByMimeType == null) {
        languagesByMimeType = ourRegisteredMimeTypes.computeIfAbsent(mimeType, (mime) -> new ArrayList<>());
      }
      languagesByMimeType.add(this);
    }

    if (baseLanguage != null) {
      baseLanguage.myDialects.add(this);
    }
  }

  /**
   * @return collection of all languages registered so far.
   */
  public static @NotNull Collection<Language> getRegisteredLanguages() {
    final Collection<Language> languages = ourRegisteredLanguages.values();
    return Collections.unmodifiableCollection(new ArrayList<>(languages));
  }

  public static void unregisterLanguages(ClassLoader classLoader) {
    List<Class<? extends Language>> classes = new ArrayList<>(ourRegisteredLanguages.keySet());
    for (Class<? extends Language> clazz : classes) {
      if (clazz.getClassLoader() == classLoader) {
        unregisterLanguage(ourRegisteredLanguages.get(clazz));
      }
    }
    IElementType.unregisterElementTypes(classLoader);
  }

  public static void unregisterLanguage(@NotNull Language language) {
    IElementType.unregisterElementTypes(language);
    ourRegisteredLanguages.remove(language.getClass());
    ourRegisteredIDs.remove(language.getID());
    for (String mimeType : language.getMimeTypes()) {
      ourRegisteredMimeTypes.remove(mimeType);
    }
    final Language baseLanguage = language.getBaseLanguage();
    if (baseLanguage != null) {
      baseLanguage.unregisterDialect(language);
    }
  }

  public void unregisterDialect(Language language) {
    myDialects.remove(language);
  }

  /**
   * @param klass {@code java.lang.Class} of the particular language. Serves key purpose.
   * @return instance of the {@code klass} language registered if any.
   */
  public static <T extends Language> T findInstance(@NotNull Class<T> klass) {
    @SuppressWarnings("unchecked") T t = (T) ourRegisteredLanguages.get(klass);
    return t;
  }

  /**
   * @param mimeType of the particular language.
   * @return collection of all languages for the given {@code mimeType}.
   */
  public static @NotNull Collection<Language> findInstancesByMimeType(@Nullable String mimeType) {
    List<Language> result = mimeType == null ? null : ourRegisteredMimeTypes.get(mimeType);
    return result == null ? Collections.emptyList() : Collections.unmodifiableCollection(result);
  }

  @Override
  public String toString() {
    return "Language: " + myID;
  }

  /**
   * Returns the list of MIME types corresponding to the language. The language MIME type is used for specifying the base language
   * of a JSP page.
   *
   * @return The list of MIME types.
   */
  public String @NotNull [] getMimeTypes() {
    return myMimeTypes;
  }

  /**
   * Returns a user-readable name of the language.
   *
   * @return the name of the language.
   */
  public @NotNull String getID() {
    return myID;
  }


  public @Nullable Language getBaseLanguage() {
    return myBaseLanguage;
  }

  public @NotNull String getDisplayName() {
    return getID();
  }

  public final boolean is(Language another) {
    return this == another;
  }

  /**
   * @return whether identifiers in this language are case-sensitive. By default, delegates to the base language (if present) or returns false (otherwise).
   */
  public boolean isCaseSensitive() {
    return myBaseLanguage != null && myBaseLanguage.isCaseSensitive();
  }

  public final boolean isKindOf(Language another) {
    Language l = this;
    while (l != null) {
      if (l.is(another)) return true;
      l = l.getBaseLanguage();
    }
    return false;
  }

  public final boolean isKindOf(@NotNull String anotherLanguageId) {
    Language l = this;
    while (l != null) {
      if (l.getID().equals(anotherLanguageId)) return true;
      l = l.getBaseLanguage();
    }
    return false;
  }

  public @NotNull List<Language> getDialects() {
    return myDialects;
  }

  public static @Nullable Language findLanguageByID(String id) {
    return id == null ? null : ourRegisteredIDs.get(id);
  }

  /**
   * Fake language identifier without registering
   */
  protected Language(@NotNull String ID, @SuppressWarnings("UnusedParameters") boolean register) {
    Language language = findLanguageByID(ID);
    if (language != null) {
      throw new IllegalArgumentException("Language with ID=" + ID + " already registered: " + language + "; " + language.getClass());
    }
    myID = ID;
    myBaseLanguage = null;
    myMimeTypes = null;
  }
}
