// Copyright 2000-2018 JetBrains s.r.o. Use of this source code is governed by the Apache 2.0 license that can be found in the LICENSE file.
package com.intellij.psi.tree;

import com.intellij.lang.Language;
import com.intellij.openapi.progress.ProgressManager;
import com.intellij.psi.ITokenSequence;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * A token type which represents a fragment of text (possibly in a different language)
 * which is not parsed during the current lexer or parser pass and can be parsed later when
 * its contents is requested.
 *
 * @author max
 */

public abstract class IReparseableElementType extends ILazyParseableElementType {

  public IReparseableElementType(@NotNull @NonNls final String debugName) {
    this(debugName, null);
  }

  public IReparseableElementType(@NotNull @NonNls final String debugName, @Nullable final Language language) {
    super(debugName, language);
  }

  /**
   * Allows to construct element types without registering them, as in {@link IElementType#IElementType(String, Language, boolean)}.
   */
  public IReparseableElementType(@NotNull @NonNls final String debugName, @Nullable final Language language, final boolean register) {
    super(debugName, language, register);
  }

  public abstract boolean isParsable(@NotNull ITokenSequence tokenSequence,
                                     int startLexeme,
                                     int lexemeCount);

  public static boolean hasProperBraceBalance(
    @NotNull ITokenSequence tokenSequence,
    int lexIndex,
    int lexCount,
    @NotNull IElementType leftBrace,
    @NotNull IElementType rightBrace) {
    if (tokenSequence.lexType(lexIndex) != leftBrace) return false;
    lexIndex++;
    lexCount--;
    var balance = 1;
    while (lexCount > 0) {
      ProgressManager.checkCanceled();
      IElementType type = tokenSequence.lexType(lexIndex);
      if (balance == 0) {
        //the last brace is not the last token
        return false;
      }

      if (type == leftBrace) {
        balance++;
      } else if (type == rightBrace) {
        balance--;
      }
      lexCount--;
      lexIndex++;
    }
    return balance == 0;
  }
}
