package com.intellij.psi.builder;

import com.intellij.lang.*;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.util.KeyWithDefaultValue;
import com.intellij.psi.ITokenSequence;
import com.intellij.psi.TokenType;
import com.intellij.psi.tree.IElementType;
import com.intellij.psi.tree.TokenSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

import static com.intellij.lang.WhitespacesBinders.DEFAULT_RIGHT_BINDER;

public abstract class FleetPsiBuilder<N> implements PsiBuilder {
  static final Logger LOG = Logger.getInstance(FleetPsiBuilder.class);

  private static class Lexeme {
    IElementType tokenType;
    int startOffset;
    int endOffset;
  }

  int myCurrentLexemeIndex;
  private final Lexeme myCurrentLexeme = new Lexeme();

  private final TokenSet myWhitespaces;
  private TokenSet myComments;

  final CharSequence myText;
  private boolean myDebugMode;
  final int myLexemeCount;
  boolean myTokenTypeChecked;
  private ITokenTypeRemapper myRemapper;
  private WhitespaceSkippedCallback myWhitespaceSkippedCallback;
  private final Map<Key<?>, Object> myMap = new HashMap<>(0);

  private IElementType myCachedTokenType;

  private final MarkerPool myPool = new MarkerPool(this);
  final MarkerOptionalData myOptionalData = new MarkerOptionalData();
  final MarkerProduction myProduction = new MarkerProduction(myPool, myOptionalData);
  final ITokenSequence myTokens;
  final int myStartLexeme;

  public FleetPsiBuilder(@NotNull CharSequence text,
                         @NotNull ITokenSequence tokens,
                         @NotNull TokenSet whitespaceTokens,
                         @NotNull TokenSet commentTokens,
                         int startLexeme,
                         int lexemeCount) {

    myText = text;
    myStartLexeme = startLexeme;
    myWhitespaces = whitespaceTokens;
    myComments = commentTokens;

    myTokens = tokens;
    myLexemeCount = lexemeCount + startLexeme;
    myCurrentLexemeIndex = startLexeme;
    updateCurrentLexeme();
  }

  @NotNull
  public abstract N getRoot();

  @Override
  public boolean isWhitespaceOrComment(@NotNull IElementType elementType) {
    return myWhitespaces.contains(elementType) || myComments.contains(elementType);
  }

  @Override
  public void enforceCommentTokens(@NotNull TokenSet tokens) {
    myComments = tokens;
  }

  @Override
  @Nullable
  public StartMarker getLatestDoneMarker() {
    int index = myProduction.size() - 1;
    while (index >= 0) {
      StartMarker marker = myProduction.getDoneMarkerAt(index);
      if (marker != null) return marker;
      --index;
    }
    return null;
  }

  @Override
  public <T> @Nullable T getUserData(@NotNull Key<T> key) {
    //noinspection unchecked
    T value = (T)myMap.get(key);
    if (value == null && key instanceof KeyWithDefaultValue) {
      value = ((KeyWithDefaultValue<T>)key).getDefaultValue();
      putUserData(key, value);
    }
    return value;
  }

  @Override
  public <T> void putUserData(@NotNull Key<T> key, @Nullable T value) {
    myMap.put(key, value);
  }


  public abstract static class ProductionMarker implements LighterASTNode {
    final int markerId;
    protected final FleetPsiBuilder<?> myBuilder;
    protected int myLexemeIndex = -1;
    protected ProductionMarker myNext;

    ProductionMarker(int markerId, @NotNull FleetPsiBuilder<?> builder) {
      this.markerId = markerId;
      myBuilder = builder;
    }

    void clean() {
      myLexemeIndex = -1;
      myNext = null;
    }

    @Override
    public int getStartOffset() {
      return myBuilder.getLexemeStart(myLexemeIndex);
    }

    public int getStartIndex() {
      return myLexemeIndex;
    }

    public int getEndIndex() {
      throw new UnsupportedOperationException("Shall not be called on this kind of markers");
    }

    @NotNull
    abstract WhitespacesAndCommentsBinder getBinder(boolean done);

    abstract void setLexemeIndex(int lexemeIndex, boolean done);

    abstract int getLexemeIndex(boolean done);
  }

  static class StartMarker extends ProductionMarker implements Marker {
    IElementType myType;
    int myDoneLexeme = -1;
    ProductionMarker myFirstChild;
    ProductionMarker myLastChild;

    StartMarker(int markerId, FleetPsiBuilder<?> builder) {
      super(markerId, builder);
    }

    @Override
    void clean() {
      super.clean();
      myBuilder.myOptionalData.clean(markerId);

      myType = null;
      myDoneLexeme = -1;
      myFirstChild = myLastChild = null;
    }

    @Override
    public int getEndOffset() {
      return myBuilder.getLexemeStart(getEndIndex());
    }

    @Override
    public int getEndIndex() {
      return myDoneLexeme;
    }

    @NotNull
    @Override
    WhitespacesAndCommentsBinder getBinder(boolean done) {
      return myBuilder.myOptionalData.getBinder(markerId, done);
    }

    @Override
    void setLexemeIndex(int lexemeIndex, boolean done) {
      if (done) myDoneLexeme = lexemeIndex;
      else myLexemeIndex = lexemeIndex;
    }

    @Override
    int getLexemeIndex(boolean done) {
      return done ? myDoneLexeme : myLexemeIndex;
    }

    public void addChild(@NotNull ProductionMarker node) {
      if (myFirstChild == null) {
        myFirstChild = node;
      } else {
        myLastChild.myNext = node;
      }
      myLastChild = node;
    }

    @NotNull
    @Override
    public Marker precede() {
      return myBuilder.precede(this);
    }

    @Override
    public void drop() {
      myBuilder.myProduction.dropMarker(this);
    }

    @Override
    public void rollbackTo() {
      myBuilder.rollbackTo(this);
    }

    @Override
    public void done(@NotNull IElementType type) {
      if (type == TokenType.ERROR_ELEMENT) {
        LOG.warn("Error elements with empty message are discouraged. Please use builder.error() instead", new RuntimeException());
      }
      myType = type;
      myBuilder.processDone(this, null, null);
    }

    @Override
    public void collapse(@NotNull IElementType type) {
      done(type);
      myBuilder.myOptionalData.markCollapsed(markerId);
    }

    @Override
    public void doneBefore(@NotNull IElementType type, @NotNull Marker before) {
      if (type == TokenType.ERROR_ELEMENT) {
        LOG.warn("Error elements with empty message are discouraged. Please use builder.errorBefore() instead", new RuntimeException());
      }
      myType = type;
      myBuilder.processDone(this, null, (StartMarker) before);
    }

    @Override
    public void doneBefore(@NotNull final IElementType type, @NotNull final Marker before, @NotNull final String errorMessage) {
      StartMarker marker = (StartMarker) before;
      ErrorItem errorItem = myBuilder.myPool.allocateErrorItem();
      errorItem.myMessage = errorMessage;
      errorItem.myLexemeIndex = marker.myLexemeIndex;
      myBuilder.myProduction.addBefore(errorItem, marker);
      doneBefore(type, marker);
    }

    @Override
    public void error(@NotNull String message) {
      myType = TokenType.ERROR_ELEMENT;
      myBuilder.processDone(this, message, null);
    }

    @Override
    public void errorBefore(@NotNull final String message, @NotNull final Marker before) {
      myType = TokenType.ERROR_ELEMENT;
      myBuilder.processDone(this, message, (StartMarker) before);
    }

    @Override
    public IElementType getTokenType() {
      return myType;
    }

    @Override
    public void setCustomEdgeTokenBinders(final WhitespacesAndCommentsBinder left, final WhitespacesAndCommentsBinder right) {
      if (left != null) {
        myBuilder.myOptionalData.assignBinder(markerId, left, false);
      }
      if (right != null) {
        myBuilder.myOptionalData.assignBinder(markerId, right, true);
      }
    }

    @Override
    public String toString() {
      if (myLexemeIndex < 0) return "<dropped>";
      boolean isDone = isDone();
      CharSequence originalText = myBuilder.getOriginalText();
      int startOffset = getStartOffset();
      int endOffset = isDone ? getEndOffset() : myBuilder.getCurrentOffset();
      CharSequence text = originalText.subSequence(startOffset, endOffset);
      return isDone ? text.toString() : text + "\u2026";
    }

    boolean isDone() {
      return myDoneLexeme != -1;
    }
  }

  @NotNull
  private Marker precede(final StartMarker marker) {
    assert marker.myLexemeIndex >= 0 : "Preceding disposed marker";
    if (myDebugMode) {
      myProduction.assertNoDoneMarkerAround(marker);
    }
    StartMarker pre = createMarker(marker.myLexemeIndex);
    myProduction.addBefore(pre, marker);
    return pre;
  }

  static class ErrorItem extends ProductionMarker {
    String myMessage;

    ErrorItem(int markerId, FleetPsiBuilder<?> builder) {
      super(markerId, builder);
    }

    @Override
    void clean() {
      super.clean();
      myMessage = null;
    }

    @NotNull
    @Override
    public WhitespacesAndCommentsBinder getBinder(boolean done) {
      assert !done;
      return DEFAULT_RIGHT_BINDER;
    }

    @Override
    void setLexemeIndex(int lexemeIndex, boolean done) {
      assert !done;
      myLexemeIndex = lexemeIndex;
    }

    @Override
    int getLexemeIndex(boolean done) {
      assert !done;
      return myLexemeIndex;
    }

    @Override
    public int getEndOffset() {
      return getStartOffset();
    }

    @Override
    public int getEndIndex() {
      return getStartIndex();
    }

    @NotNull
    @Override
    public IElementType getTokenType() {
      return TokenType.ERROR_ELEMENT;
    }
  }

  @NotNull
  @Override
  public CharSequence getOriginalText() {
    return myText;
  }

  @Override
  @Nullable
  public IElementType getTokenType() {
    IElementType cached = myCachedTokenType;
    if (cached == null) {
      myCachedTokenType = cached = calcTokenType();
    }
    return cached;
  }

  void clearCachedTokenType() {
    myCachedTokenType = null;
  }

  private IElementType remapCurrentToken() {
    if (myCachedTokenType != null) return myCachedTokenType;
    if (myRemapper != null) {
      remapCurrentToken(myRemapper.filter(myCurrentLexeme.tokenType, myCurrentLexeme.startOffset,
        myCurrentLexeme.endOffset, myText));
    }
    return myCurrentLexeme.tokenType;
  }

  private IElementType calcTokenType() {
    if (eof()) return null;

    if (myRemapper != null) {
      //remaps current token, and following, which remaps to spaces and comments
      skipWhitespace();
    }
    return myCurrentLexeme.tokenType;
  }

  private IElementType getLexType(int index) {
    return myTokens.lexType(index);
  }

  @Override
  public void setTokenTypeRemapper(ITokenTypeRemapper remapper) {
    myRemapper = remapper;
    myTokenTypeChecked = false;
    clearCachedTokenType();
  }

  void updateCurrentLexeme() {
    if (myCurrentLexemeIndex < myLexemeCount) {
      myCurrentLexeme.tokenType = myTokens.lexType(myCurrentLexemeIndex);
      myCurrentLexeme.startOffset = myTokens.lexStart(myCurrentLexemeIndex);
      myCurrentLexeme.endOffset = myTokens.lexStart(myCurrentLexemeIndex + 1);
    } else {
      myCurrentLexeme.tokenType = TokenType.BAD_CHARACTER;
      myCurrentLexeme.startOffset = myTokens.lexStart(myCurrentLexemeIndex);
      myCurrentLexeme.endOffset = myCurrentLexeme.startOffset;
    }
  }

  @Override
  public void remapCurrentToken(IElementType type) {
    if (myCurrentLexemeIndex < myLexemeCount) {
      // IJ adds new lexeme at the end of the list, but doesn't change the lexeme count
      // we'll simply ignore this request
      myTokens.remap(myCurrentLexemeIndex, type);
      updateCurrentLexeme();
      clearCachedTokenType();
    }
  }

  @Nullable
  @Override
  public IElementType lookAhead(int steps) {
    int cur = shiftOverWhitespaceForward(myCurrentLexemeIndex);

    while (steps > 0) {
      cur = shiftOverWhitespaceForward(cur + 1);
      steps--;
    }

    return cur < myLexemeCount ? getLexType(cur) : null;
  }

  private int shiftOverWhitespaceForward(int lexIndex) {
    while (lexIndex < myLexemeCount && whitespaceOrComment(getLexType(lexIndex))) {
      lexIndex++;
    }
    return lexIndex;
  }

  @Override
  public IElementType rawLookup(int steps) {
    int cur = myCurrentLexemeIndex + steps;
    return cur < myLexemeCount && cur >= 0 ? getLexType(cur) : null;
  }

  @Override
  public int rawTokenTypeStart(int steps) {
    int cur = myCurrentLexemeIndex + steps;
    if (cur < 0) return -1;
    if (cur >= myLexemeCount) return getOriginalText().length();
    return getLexemeStart(cur);
  }

  @Override
  public int rawTokenIndex() {
    return myCurrentLexemeIndex;
  }

  @Override
  public void setWhitespaceSkippedCallback(@Nullable final WhitespaceSkippedCallback callback) {
    myWhitespaceSkippedCallback = callback;
  }

  @Override
  public void advanceLexer() {
    if (eof()) return;

    myTokenTypeChecked = false;
    myCurrentLexemeIndex++;
    updateCurrentLexeme();
    clearCachedTokenType();
  }

  private void skipWhitespace() {
    while (myCurrentLexemeIndex < myLexemeCount && whitespaceOrComment(remapCurrentToken())) {
      onSkip(myCurrentLexeme.tokenType, myCurrentLexeme.startOffset,
        myCurrentLexemeIndex + 1 < myLexemeCount ? myCurrentLexeme.endOffset : myText.length());
      myCurrentLexemeIndex++;
      updateCurrentLexeme();
      clearCachedTokenType();
    }
  }

  private void onSkip(IElementType type, int start, int end) {
    if (myWhitespaceSkippedCallback != null) {
      myWhitespaceSkippedCallback.onSkip(type, start, end);
    }
  }

  @Override
  public int getCurrentOffset() {
    if (eof()) return getOriginalText().length();
    return myCurrentLexeme.startOffset;
  }

  private int getLexemeStart(int idx) {
    return myTokens.lexStart(idx);
  }

  @Override
  @Nullable
  public String getTokenText() {
    if (eof()) return null;
    final IElementType type = getTokenType();
    if (type instanceof TokenWrapper) {
      return ((TokenWrapper) type).getValue();
    }
    return myText.subSequence(myCurrentLexeme.startOffset, myCurrentLexeme.endOffset).toString();
  }

  public boolean whitespaceOrComment(IElementType token) {
    return myWhitespaces.contains(token) || myComments.contains(token);
  }

  @NotNull
  @Override
  public Marker mark() {
    if (!myProduction.isEmpty()) {
      skipWhitespace();
    }

    StartMarker marker = createMarker(myCurrentLexemeIndex);
    myProduction.addMarker(marker);
    return marker;
  }

  @NotNull
  private StartMarker createMarker(final int lexemeIndex) {
    StartMarker marker = myPool.allocateStartMarker();
    marker.myLexemeIndex = lexemeIndex;
    if (myDebugMode) {
      myOptionalData.notifyAllocated(marker.markerId);
    }
    return marker;
  }

  @Override
  public final boolean eof() {
    if (!myTokenTypeChecked) {
      myTokenTypeChecked = true;
      skipWhitespace();
    }
    return myCurrentLexemeIndex >= myLexemeCount;
  }

  private void rollbackTo(@NotNull StartMarker marker) {
    assert marker.myLexemeIndex >= 0 : "The marker is already disposed";
    if (myDebugMode) {
      myProduction.assertNoDoneMarkerAround(marker);
    }
    myCurrentLexemeIndex = marker.myLexemeIndex;
    updateCurrentLexeme();
    myTokenTypeChecked = true;
    myProduction.rollbackTo(marker);
    clearCachedTokenType();
  }

  private void processDone(@NotNull StartMarker marker, @Nullable String errorMessage, @Nullable StartMarker before) {
    doValidityChecks(marker, before);

    if (errorMessage != null) {
      myOptionalData.setErrorMessage(marker.markerId, errorMessage);
    }

    int doneLexeme = before == null ? myCurrentLexemeIndex : before.myLexemeIndex;
    if (marker.myType.isLeftBound() && isEmpty(marker.myLexemeIndex, doneLexeme)) {
      marker.setCustomEdgeTokenBinders(DEFAULT_RIGHT_BINDER, null);
    }
    marker.myDoneLexeme = doneLexeme;
    myProduction.addDone(marker, before);
  }

  private boolean isEmpty(final int startIdx, final int endIdx) {
    for (int i = startIdx; i < endIdx; i++) {
      final IElementType token = getLexType(i);
      if (!whitespaceOrComment(token)) return false;
    }
    return true;
  }

  private void doValidityChecks(@NotNull StartMarker marker, @Nullable StartMarker before) {
    if (marker.isDone()) {
      LOG.error("Marker already done.");
    }

    if (myDebugMode) {
      myProduction.doHeavyChecksOnMarkerDone(marker, before);
    }
  }

  @Override
  public void error(@NotNull String messageText) {
    ProductionMarker lastMarker = myProduction.getStartMarkerAt(myProduction.size() - 1);
    if (lastMarker instanceof ErrorItem && lastMarker.myLexemeIndex == myCurrentLexemeIndex) {
      return;
    }
    ErrorItem marker = myPool.allocateErrorItem();
    marker.myMessage = messageText;
    marker.myLexemeIndex = myCurrentLexemeIndex;
    myProduction.addMarker(marker);
  }

  private static final String UNBALANCED_MESSAGE =
    "Unbalanced tree. Most probably caused by unbalanced markers. " +
      "Try calling setDebugMode(true) against PsiBuilder passed to identify exact location of the problem";


  void assertMarkersBalanced(boolean condition, @Nullable ProductionMarker marker) {
    if (condition) return;

    reportUnbalancedMarkers(marker);
  }

  private void reportUnbalancedMarkers(@Nullable ProductionMarker marker) {
    int index = marker != null ? marker.getStartIndex() + 1 : myTokens.getLexemeCount();
    CharSequence context =
      index < myTokens.getLexemeCount() ? myText.subSequence(Math.max(0, getLexemeStart(index) - 1000), getLexemeStart(index)) : "<none>";
    LOG.error(UNBALANCED_MESSAGE + "\ncontext: '" + context + "'");
  }

  void balanceWhiteSpaces() {
    RelativeTokenTypesView wsTokens = new RelativeTokenTypesView();
    RelativeTokenTextView tokenTextGetter = new RelativeTokenTextView();
    int lastIndex = 0;

    for (int i = 1, size = myProduction.size() - 1; i < size; i++) {
      ProductionMarker starting = myProduction.getStartMarkerAt(i);
      if (starting instanceof StartMarker) {
        assertMarkersBalanced(((StartMarker) starting).isDone(), starting);
      }
      boolean done = starting == null;
      ProductionMarker item = starting != null ? starting : Objects.requireNonNull(myProduction.getDoneMarkerAt(i));

      WhitespacesAndCommentsBinder binder = item.getBinder(done);
      int lexemeIndex = item.getLexemeIndex(done);

      boolean recursive = binder instanceof WhitespacesAndCommentsBinder.RecursiveBinder;
      int prevProductionLexIndex = recursive ? 0 : myProduction.getLexemeIndexAt(i - 1);
      int wsStartIndex = Math.max(lexemeIndex, lastIndex);
      while (wsStartIndex > prevProductionLexIndex && whitespaceOrComment(getLexType(wsStartIndex - 1))) wsStartIndex--;

      int wsEndIndex = shiftOverWhitespaceForward(lexemeIndex);

      if (wsStartIndex != wsEndIndex) {
        wsTokens.configure(wsStartIndex, wsEndIndex);
        tokenTextGetter.configure(wsStartIndex);
        boolean atEnd = wsStartIndex == 0 || wsEndIndex == myLexemeCount;
        lexemeIndex = wsStartIndex + binder.getEdgePosition(wsTokens, atEnd, tokenTextGetter);
        item.setLexemeIndex(lexemeIndex, done);
        if (recursive) {
          myProduction.confineMarkersToMaxLexeme(i, lexemeIndex);
        }
      } else if (lexemeIndex < wsStartIndex) {
        lexemeIndex = wsStartIndex;
        item.setLexemeIndex(wsStartIndex, done);
      }

      lastIndex = lexemeIndex;
    }
  }

  private final class RelativeTokenTypesView extends AbstractList<IElementType> {
    private int myStart;
    private int mySize;

    private void configure(int start, int end) {
      myStart = start;
      mySize = end - start;
    }

    @Override
    public IElementType get(int index) {
      return getLexType(myStart + index);
    }

    @Override
    public int size() {
      return mySize;
    }
  }

  private final class RelativeTokenTextView implements WhitespacesAndCommentsBinder.TokenTextGetter {
    private int myStart;

    private void configure(int start) {
      myStart = start;
    }

    @Override
    @NotNull
    public CharSequence get(int i) {
      int idx = myStart + i;
      return myText.subSequence(myTokens.lexStart(idx), myTokens.lexStart(idx + 1));
    }
  }


  @Nullable
  public static String getErrorMessage(@NotNull LighterASTNode node) {
    if (node instanceof ErrorItem errorNode) return errorNode.myMessage;
    if (node instanceof final StartMarker marker) {
      if (marker.myType == TokenType.ERROR_ELEMENT) {
        return marker.myBuilder.myOptionalData.getDoneError(marker.markerId);
      }
    }

    return null;
  }

  @Override
  public void setDebugMode(boolean dbgMode) {
    myDebugMode = dbgMode;
  }

  @Nullable
  public List<ProductionMarker> getProductions() {
    return new AbstractList<>() {
      @Override
      public ProductionMarker get(int index) {
        return myProduction.getMarkerAt(index);
      }

      @Override
      public int size() {
        return myProduction.size();
      }
    };
  }
}
