package org.aya.gradle;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.concurrent.TimeUnit;

public interface BuildUtil {
  static String gitRev(File rootDir) throws IOException, InterruptedException {
    var proc = new ProcessBuilder("git", "rev-parse", "HEAD")
      .directory(rootDir)
      .start();
    proc.waitFor(2, TimeUnit.SECONDS);
    var stdout = proc.getInputStream().readAllBytes();
    var stderr = proc.getErrorStream().readAllBytes();
    if (stdout.length == 0) {
      throw new IOException(new String(stderr));
    }
    return new String(stdout).trim();
  }

  String OLD_EX = "java/lang/MatchException";
  String NEW_EX = "java/lang/RuntimeException";

  int CONSTANT_Utf8 = 1;
  int CONSTANT_Long = 5;
  int CONSTANT_Double = 6;
  int SIZE_INVALID = -114514;
  int U1 = 1;
  int U2 = 2;
  int U4 = 4;

  // Mapping from constant pool entry tag -> entry size without the tag (0 means variable).
  int[] POOL_ENTRY_SIZE = new int[]{
    /* ........................... =  0 */ SIZE_INVALID,
    /* CONSTANT_Utf8               =  1 */ 0,
    /* ..........................  =  2 */ SIZE_INVALID,
    /* CONSTANT_Integer            =  3 */ U4,
    /* CONSTANT_Float              =  4 */ U4,
    /* CONSTANT_Long               =  5 */ U4 + U4,
    /* CONSTANT_Double             =  6 */ U4 + U4,
    /* CONSTANT_Class              =  7 */ U2,
    /* CONSTANT_String             =  8 */ U2,
    /* CONSTANT_Fieldref           =  9 */ U2 + U2,
    /* CONSTANT_Methodref          = 10 */ U2 + U2,
    /* CONSTANT_InterfaceMethodref = 11 */ U2 + U2,
    /* CONSTANT_NameAndType        = 12 */ U2 + U2,
    /* ........................... = 13 */ SIZE_INVALID,
    /* ........................... = 14 */ SIZE_INVALID,
    /* CONSTANT_MethodHandle       = 15 */ U1 + U2,
    /* CONSTANT_MethodType         = 16 */ U2,
    /* ........................... = 17 */ SIZE_INVALID,
    /* CONSTANT_InvokeDynamic      = 18 */ U2 + U2,
    /* CONSTANT_Module             = 19 */ U2,
    /* CONSTANT_Package            = 20 */ U2,
  };

  static void stripPreview(Path root, Path classFile) throws IOException {
    stripPreview(root, classFile, NEW_EX);
  }

  static void stripPreview(Path root, Path classFile, String matchException) throws IOException {
    stripPreview(root, classFile, true, true, matchException);
  }

  static void stripPreview(Path root, Path classFile, boolean forceJava17, boolean verbose, String matchException) throws IOException {
    var relative = root.relativize(classFile.toAbsolutePath());

    // struct Head {
    //   u4 magic;
    //   u2 minor_version;
    //   u2 major_version;
    //   u2 constant_pool_count;
    //   CP constant_pool_entries[constant_pool_count - 1];
    //   ...
    // };
    var mm = ByteBuffer.wrap(Files.readAllBytes(classFile));
    int magic = mm.getInt(0);
    if (magic != 0xCAFEBABE) return;

    var time = Files.getLastModifiedTime(classFile);
    var tmpFile = classFile.resolveSibling(classFile.getFileName() + ".tmp.class");
    try (var raf = new RandomAccessFile(tmpFile.toFile(), "rw")) {
      raf.writeInt(magic);

      int minor = mm.getShort(4) & 0xFFFF;
      if (minor == 0xFFFF) {
        log(verbose, "AyaBuild[minor] %s: 0x%x -> 0", relative, minor);
        raf.writeShort(0);
      } else {
        raf.writeShort(minor);
      }

      int major = mm.getShort(6) & 0xFFFF;
      // Java 17 uses major version 61
      if (forceJava17 && major > 61) {
        log(verbose, "AyaBuild[major] %s: %d (Java %d) -> 61 (Java 17)", relative, major, 17 + major - 61);
        raf.writeShort(61);
        int poolEnd = replaceException(relative, verbose, mm, raf, matchException);
        raf.write(mm.array(), poolEnd, mm.capacity() - poolEnd);
      } else {
        raf.writeShort(major);
        raf.write(mm.array(), 8, mm.capacity() - 8);
      }
    }
    // or we may lose incremental compilation
    Files.move(tmpFile, classFile, StandardCopyOption.REPLACE_EXISTING);
    Files.setLastModifiedTime(classFile, time);
  }

  private static int replaceException(Path relative, boolean verbose, ByteBuffer mm, RandomAccessFile raf, String matchException) throws IOException {
    final int poolCount = 8;
    final int poolStart = 10;
    // struct CP {
    //   u1 tag;
    // };
    // struct CONSTANT_Utf8_info : struct CP {
    //   u2 length;
    //   u1 bytes[length];
    // };
    int cpCount = mm.getShort(poolCount) & 0xFFFF;
    raf.writeShort(cpCount);

    int off = poolStart;
    // CP[0] is unused.
    for (int i = 1; i < cpCount; i++) {
      int tag = mm.get(off++) & 0xFF;
      raf.writeByte(tag);

      if (tag != CONSTANT_Utf8) {
        if (tag == 0 || tag >= POOL_ENTRY_SIZE.length) throw new IllegalStateException(
          "AyaBuild[cpool] %s: unexpected pool entry tag %d at pool index %d"
            .formatted(relative, tag, i));

        int length = POOL_ENTRY_SIZE[tag];
        if (length == SIZE_INVALID) throw new IllegalStateException(
          "AyaBuild[cpool] %s: unexpected pool entry length %d at pool index %d with tag %d"
            .formatted(relative, length, i, tag));

        raf.write(mm.array(), off, length);
        off += length;

        // It takes two.
        if (tag == CONSTANT_Long || tag == CONSTANT_Double)
          i += 1;
        continue;
      }

      // Got the string to patch
      int length = mm.getShort(off) & 0xFFFF;
      off += 2;

      var bytes = new byte[length];
      mm.get(off, bytes);
      off += length;

      var string = new String(bytes, StandardCharsets.UTF_8);
      if (string.equals(OLD_EX)) {
        log(verbose, "AyaBuild[cpool] %s: %s(%d) -> %s", relative, OLD_EX, i, matchException);
        raf.writeShort(matchException.length());
        raf.write(matchException.getBytes(StandardCharsets.UTF_8));
      } else {
        raf.writeShort(length);
        raf.write(bytes);
      }
    }

    return off;
  }

  private static void log(boolean verbose, String fmt, Object... args) {
    if (verbose) System.out.printf(fmt + "%n", args);
  }
}
