/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.test.rest.client.person.controllers;

import org.autumnframework.service.test.api.PersonDTO;
import org.autumnframework.service.test.services.PersonService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.web.bind.annotation.*;

import java.util.List;
import java.util.Optional;
import java.util.UUID;

/**
 * Test controller that proxies the personService we try to test. So:
 *
 * TestPersonAPIController -> PersonAPIService -> PersonAPIRepository(extends DefaultAPIRepository)
 */
@RestController
@RequestMapping("/test/persons")
public class TestPersonAPIController {

    private final PersonService personService;

    @Autowired
    public TestPersonAPIController(PersonService personService) {
        this.personService = personService;
    }

    @PostMapping("/")
    @ResponseBody
    public PersonDTO create(@RequestBody PersonDTO dto) {
        return this.personService.create(dto);
    }


    @GetMapping("/{id}")
    public Optional<PersonDTO> findById(@PathVariable("id") UUID id) {
        return this.personService.findById(id);
    }

    @GetMapping("/ids")
    public List<PersonDTO> findByIds(@RequestParam(name = "id") List<UUID> ids) {
        return this.personService.findByIds(ids);
    }

    @GetMapping("/")
    public Page<PersonDTO> findAll(Pageable pageable) {
        return this.personService.findAll(pageable);
    }

    @PutMapping(path = "/")
    @ResponseBody
    public PersonDTO update(@RequestBody PersonDTO dto) {
        return this.personService.update(dto);
    }

    @DeleteMapping("/{id}")
    public void delete(@PathVariable("id") UUID id) {
        this.personService.deleteById(id);
    }

    @DeleteMapping("/ids")
    public void deleteByIds(@RequestParam(name = "id") List<UUID> ids) {
        this.personService.deleteByIds(ids);
    }

    @Scheduled(cron = "0/10 * * * * *")
    public void schedule() {
        this.personService.findAll(PageRequest.of(0, 10));
    }
}
