/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.service.csv.server.controllers.elementary.helper;

import java.io.IOException;
import java.io.Writer;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;

import org.apache.commons.lang3.ArraySorter;
import org.apache.commons.lang3.ClassUtils;
import org.apache.commons.lang3.builder.ToStringExclude;

import com.google.common.base.CaseFormat;

import jakarta.persistence.Id;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.autumnframework.service.api.dtos.Identifiable;
import org.autumnframework.service.jpa.entities.ApiEntity;

/**
 * Used in the csv rest api exporter / export streamer
 */
@Slf4j
public class ToStringHelper {
    
    @Getter
    List<Field> fields;
    
    @Getter
    List<String> excludeFields;
    
    boolean camelCaseHeader = true;
    
    /**
     * @param clazz
     * @param excludeFields
     */
    public ToStringHelper(Class<?> clazz, List<String> excludeFields) {
        this.excludeFields = excludeFields;
        this.fields = getFieldsFromClass(clazz);
    }
    
    /**
     * @param clazz
     * @param excludeFields
     * @param camelCaseHeader
     */
    public ToStringHelper(Class<?> clazz, List<String> excludeFields, boolean camelCaseHeader) {
        this.excludeFields = excludeFields;
        this.fields = getFieldsFromClass(clazz);
        this.camelCaseHeader = camelCaseHeader;
    }
    
    /**
     * @param clazz
     * @return
     */
    private List<Field> getFieldsFromClass(final Class<?> clazz) {
        List<Field> list = new ArrayList<>();
        final Field[] fields = ArraySorter.sort(clazz.getDeclaredFields(), Comparator.comparing(Field::getName));
        AccessibleObject.setAccessible(fields, true);
        for (final Field field : fields) {
           if (this.accept(field)) {
               list.add(field);
           }
        }
        return list;
    }
    
    protected boolean accept(final Field field) {
        if (field.getName().indexOf(ClassUtils.INNER_CLASS_SEPARATOR_CHAR) != -1) {
            // Reject field from inner class.
            return false;
        }
        if (Modifier.isTransient(field.getModifiers()) || Modifier.isStatic(field.getModifiers())) {
            // Reject transient and static fields.
            return false;
        }
        if (field.getAnnotation(Id.class) != null) {
            // Reject primary keys
            return false;
        }
        if (this.excludeFields != null && this.excludeFields.contains(field.getName())) {
            // Skip excluded fields
            return false;
        }
        return !field.isAnnotationPresent(ToStringExclude.class);
    }

    /**
     * @param <T>
     * @param objects
     * @param out
     */
    public <T> void writeObjects(final List<T> objects, Writer out) {
        objects.stream().forEach(o -> { try {
            writeObject(o, out);
        } catch (IllegalArgumentException | IllegalAccessException | IOException e) {
            log.error(e.getMessage(), e);
        }});
    }
    
    /**
     * @param <T>
     * @param object
     * @param out
     * @throws IllegalArgumentException
     * @throws IllegalAccessException
     * @throws IOException
     */
    public <T> void writeObject(final T object, Writer out) throws IllegalArgumentException, IllegalAccessException, IOException {
        StringBuffer buf = new StringBuffer();
        boolean first = true;
        for (final Field field : fields) {
            String value = addQuote(getAndStringifyValue(field, object));
            if (!first) {
                buf.append(",");
            } else {
                first = false;
            }
            buf.append(value);
        }
        buf.append("\n");
        out.write(buf.toString());
    }
        
    private String getAndStringifyValue(Field field, Object object) throws IllegalArgumentException, IllegalAccessException {
        Object val = field.get(object);
        return getValue(val);
    }
    
    private String getValue(Object val) {
        if (val == null) {
            return null;
        }
        if (val instanceof Identifiable) {
            return ((Identifiable)val).getId().toString();
        }
        if (val instanceof ApiEntity) {
            return ((ApiEntity)val).getApiId().toString();
        }
        if (val instanceof Collection<?>) {
            StringBuffer buffer = new StringBuffer();
            boolean first = true;
            for (Object obj : (Collection<?>)val) {
                if (!first) {
                    buffer.append(",");
                } else {
                    first = false;
                }
                buffer.append(getValue(obj));
            }
            return buffer.toString();
        }
        if (val instanceof LocalDateTime) {
            return ((LocalDateTime)val).format(DateTimeFormatter.ISO_LOCAL_DATE_TIME);
        }
        if (val instanceof LocalDate) {
            return ((LocalDate)val).format(DateTimeFormatter.ISO_LOCAL_DATE);
        }
        return val.toString();
    }

    /**
     * @param pValue
     * @return
     */
    public static String addQuote(
            String pValue) {
        if (pValue == null) {
            return null;
        }
        if (pValue.contains("\"")) {
            pValue = pValue.replace("\"", "\"\"");
        }
        if (pValue.contains(",")
                || pValue.contains("\n")
                || pValue.contains("'")
                || pValue.contains("\\")
                || pValue.contains("\"")) {
            return "\"" + pValue + "\"";
        }
        return pValue;
    }

    /**
     * @param out
     * @throws IOException
     */
    public void writeHeader(Writer out) throws IOException {
        StringBuffer buf = new StringBuffer();
        boolean first = true;
        for (final Field field : fields) {
            String value = addQuote(properCase(field.getName()));
            if (!first) {
                buf.append(",");
            } else {
                first = false;
            }
            buf.append(value);
        }
        buf.append("\n");
        out.write(buf.toString());
    }

    private String properCase(String name) {
        return this.camelCaseHeader ? name : CaseFormat.LOWER_CAMEL.to(CaseFormat.LOWER_UNDERSCORE, name);
    }
}
