/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.service.csv.server.controllers.elementary;

import java.io.IOException;
import java.io.Writer;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.List;
import java.util.function.Function;

import org.apache.commons.collections4.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Pageable;
import org.springframework.http.HttpHeaders;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestParam;

import jakarta.servlet.http.HttpServletResponse;
import org.autumnframework.service.api.dtos.Identifiable;
import org.autumnframework.service.csv.server.controllers.elementary.helper.ToStringHelper;
import org.autumnframework.service.jpa.entities.ApiEntity;
import org.autumnframework.service.jpa.services.FullApiService;
import org.autumnframework.service.mappers.DtoEntityMapper;

/**
 * @param <DTO>
 * @param <T>
 */
public interface CsvDownloadController<DTO extends Identifiable, T extends ApiEntity> {
    
    /**
     * Logger
     */
    Logger log = LoggerFactory.getLogger(CsvDownloadController.class);

    /**
     * @return
     */
    FullApiService<T> getService();
    
    /**
     * @return
     */
    DtoEntityMapper<DTO, T> getMapper();

    /**
     * Construct a csv of all the data the findAllService can supply
     * @param page 
     * @param camelCaseHeader 
     * @param response 
     * @throws IOException 
     *
     */
    @GetMapping(path = "", produces = "text/csv")
    default void downloadDtoCsv(Pageable page, @RequestParam(name = "camelCaseHeader", defaultValue = "true") boolean camelCaseHeader, HttpServletResponse response) throws IOException {
        if (page == null) {
            page = Pageable.unpaged();
        }
        this.returnCsv(getDtoCsvContent(page), response, camelCaseHeader);
    }


    /**
     * @param date
     * @param camelCaseHeader
     * @param page
     * @param response
     * @throws IOException
     */
    @GetMapping(path = "/since/{date}", produces = "text/csv")
    default void downloadDtoCsv(@PathVariable(name = "date", required = true) LocalDateTime date, @RequestParam(name = "camelCaseHeader", defaultValue = "true") boolean camelCaseHeader, Pageable page, HttpServletResponse response) throws IOException {
        if (page == null) {
            page = Pageable.unpaged();
        }
        this.returnCsv(getDtoCsvContent(date, page), response, camelCaseHeader);
    }
    
    /**
     * @param start
     * @param end
     * @param camelCaseHeader
     * @param page
     * @param response
     * @throws IOException
     */
    @GetMapping(path = "/between/{start}/{end}", produces = "text/csv")
    default void downloadDtoCsv(@PathVariable(name = "start", required = true) LocalDateTime start, @PathVariable(name = "end", required = true) LocalDateTime end, @RequestParam(name = "camelCaseHeader", defaultValue = "true") boolean camelCaseHeader, Pageable page, HttpServletResponse response) throws IOException {
        if (page == null) {
            page = Pageable.unpaged();
        }
        this.returnCsv(getDtoCsvContent(start, end, page), response, camelCaseHeader);
    }
    
    /**
     * @param records
     * @param response
     * @param camelCaseHeader
     * @throws IOException
     */
    default void returnCsv(List<DTO> records, HttpServletResponse response, boolean camelCaseHeader) throws IOException {
        if (CollectionUtils.isNotEmpty(records)) {
            ToStringHelper helper = new ToStringHelper(records.get(0).getClass(), this.getDtoExcludeFieldNames(), camelCaseHeader);
            String csvName = getDtoCsvName(records);
            response.setContentType("text/csv; charset=utf-8");
            response.addHeader(HttpHeaders.CONTENT_DISPOSITION, String.format("attachment ; filename = \"%s.csv\"", csvName));
            Writer writer = response.getWriter();
            helper.writeHeader(writer);
            helper.writeObjects(records, writer);
        }
    }
    
    /**
     * Override this to supply your own name for the csv output
     * @return 
     */
    default String getDtoCsvName() {
        return null;
    }
    
    /**
     * Override this to skip fields in the csv output
     * @return 
     */
    default List<String> getDtoExcludeFieldNames() {
        return Collections.emptyList();
    }
    
    /**
     * Override this to supply the contents for the csv
     * @param page 
     * @return 
     */   
    default List<DTO> getDtoCsvContent(Pageable page) {
        return getService().findAll(page).map(getCsvMappingFunction()::apply).getContent();
    }
    
    /**
     * Override this to supply the contents for the csv with date selection
     * @param date 
     * @param page 
     * @return 
     */   
    default List<DTO> getDtoCsvContent(LocalDateTime date, Pageable page) {
        return getService().findAllModifiedSince(date, page).map(getCsvMappingFunction()::apply).getContent();
    }
    
    /**
     * Override this to supply the contents for the csv with date selection
     * @param start 
     * @param end 
     * @param page 
     * @return 
     */   
    default List<DTO> getDtoCsvContent(LocalDateTime start, LocalDateTime end, Pageable page) {
        return getService().findAllModifiedBetween(start, end, page).map(getCsvMappingFunction()::apply).getContent();
    }
    
    /**
     * Override this to supply a mapping function other than mapToDTO
     * @return 
     */ 
    default Function<T, DTO> getCsvMappingFunction() {
        return this.getMapper()::mapToDTO;
    }
    
    /**
     * @param records
     * @return
     */
    default String getDtoCsvName(List<DTO> records) {
        String name = getDtoCsvName();
        if (name == null && CollectionUtils.isNotEmpty(records)) {
            return records.get(0).getClass().getSimpleName();
        }
        return "output";
    }

}
