/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.service.csv.server.controllers.elementary;

import java.io.IOException;
import java.io.Writer;
import java.time.LocalDateTime;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.MediaType;
import org.springframework.lang.NonNull;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestParam;

import jakarta.persistence.EntityManager;
import jakarta.servlet.http.HttpServletResponse;
import lombok.Data;
import org.autumnframework.service.api.dtos.AuditedIdentifiable;
import org.autumnframework.service.csv.server.controllers.elementary.helper.ToStringHelper;
import org.autumnframework.service.jpa.entities.ApiEntity;

/**
 * @param <DTO>
 * @param <T>
 */
public interface AuditedCsvDownloadController<DTO extends AuditedIdentifiable, T extends ApiEntity> extends CsvDownloadController<DTO, T>{
    
    /**
     * 
     */
    Logger log = LoggerFactory.getLogger(AuditedCsvDownloadController.class);

    /**
     * @return
     */
    EntityManager getEntityManager();
    
    /**
     * @param offset
     * @param camelCaseHeader
     * @param response
     * @throws IOException
     */
    @GetMapping(path = "/stream", produces = MediaType.TEXT_EVENT_STREAM_VALUE)
    default void streamDtoCsv(@RequestParam(name = "offset", required = false, defaultValue = "0") int offset, @RequestParam(name = "camelCase", required = false, defaultValue = "true") boolean camelCaseHeader, HttpServletResponse response) throws IOException {
        this.streamCsv(getDtoCsvContentStream(), response, offset, camelCaseHeader);
    }
    
    /**
     * @param date
     * @param offset
     * @param camelCaseHeader
     * @param response
     * @throws IOException
     */
    @GetMapping(path = "/stream/since/{date}", produces = MediaType.TEXT_EVENT_STREAM_VALUE)
    default void streamDtoCsv(@PathVariable(name = "date", required = true) LocalDateTime date, @RequestParam(name = "offset", required = false, defaultValue = "0") int offset, @RequestParam(name = "camelCaseHeader", required = false, defaultValue = "true") boolean camelCaseHeader, HttpServletResponse response) throws IOException {
        this.streamCsv(getDtoCsvContentStream(date), response, offset, camelCaseHeader);
    }
    
    /**
     * @param start
     * @param end
     * @param offset
     * @param camelCaseHeader
     * @param response
     * @throws IOException
     */
    @GetMapping(path = "/stream/between/{start}/{end}", produces = MediaType.TEXT_EVENT_STREAM_VALUE)
    default void streamDtoCsv(@PathVariable(name = "start", required = true) LocalDateTime start, @PathVariable(name = "end", required = true) LocalDateTime end, @RequestParam(name = "offset", required = false, defaultValue = "0") int offset, @RequestParam(name = "camelCaseHeader", required = false, defaultValue = "true") boolean camelCaseHeader, HttpServletResponse response) throws IOException {
        this.streamCsv(getDtoCsvContentStream(start, end), response, offset, camelCaseHeader);
    }

    /**
     * @return
     */
    default Stream<T> getDtoCsvContentStream() {
        return getService().streamAll();
    }
    
    /**
     * @param date
     * @return
     */
    default Stream<T> getDtoCsvContentStream(@NonNull LocalDateTime date) {
        return getService().getRepository().streamAllByLastModifiedDateAfterOrderByCreationDate(date);
    }
    
    /**
     * @param start
     * @param end
     * @return
     */
    default Stream<T> getDtoCsvContentStream(@NonNull LocalDateTime start, @NonNull  LocalDateTime end) {
        return getService().getRepository().streamAllByLastModifiedDateBetweenOrderByCreationDate(start, end);
    }
    
    /**
     * @param records
     * @param response
     * @param offset
     * @param camelCaseHeader
     * @throws IOException
     */
    default void streamCsv(Stream<T> records, HttpServletResponse response, int offset, boolean camelCaseHeader) throws IOException {
        Writer writer = response.getWriter();
        StreamHelper<DTO> helper = new StreamHelper<>();
        helper.setWriter(writer);
        records.map(this::mapAndDetach).skip(offset).forEach(record -> {
            if (helper.getHelper() == null) {
                helper.setHelper(new ToStringHelper(record.getClass(), this.getDtoExcludeFieldNames(), camelCaseHeader));
                response.setContentType(MediaType.TEXT_EVENT_STREAM_VALUE + "; charset=utf-8");
                helper.writeHeader();
            }
            helper.writeObject(record);
        });
    }
    
    /**
     * @param entity
     * @return
     */
    default DTO mapAndDetach(T entity) {
        DTO dto = this.getCsvMappingFunction().apply(entity);
        this.getEntityManager().detach(entity);
        return dto;
    }
    
    /**
     * @param <DTO>
     */
    @Data
    public static class StreamHelper<DTO> {
        ToStringHelper helper = null;
        Writer writer = null;
        /**
         * 
         */
        public void writeHeader() {
            try {
                this.helper.writeHeader(this.writer);
            } catch (IOException ioe) {
                log.error(ioe.getMessage(), ioe);
            }
        }
        /**
         * @param d
         */
        public void writeObject(DTO d) {
            try {
                this.helper.writeObject(d, this.writer);
            } catch (IOException|IllegalAccessException ioe) {
                log.error(ioe.getMessage(), ioe);
            }
        }
    }
}
