/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.service.rabbit.client.listener;

import static org.springframework.amqp.support.converter.AbstractJavaTypeMapper.DEFAULT_CLASSID_FIELD_NAME;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;

import org.springframework.amqp.core.Message;
import org.springframework.amqp.rabbit.annotation.RabbitListener;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import io.micrometer.core.annotation.Timed;
import lombok.extern.slf4j.Slf4j;
import org.autumnframework.service.event.metadata.EventMetaData;
import org.autumnframework.service.queue.api.client.listener.CrudQueueListener;
import org.autumnframework.service.queue.api.client.listener.OnQueueCreateListener;
import org.autumnframework.service.queue.api.client.listener.OnQueueDeleteListener;
import org.autumnframework.service.queue.api.client.listener.OnQueueUpdateListener;
import org.autumnframework.service.queue.api.client.listener.batch.CrudQueueBatchListener;
import org.autumnframework.service.queue.api.client.listener.batch.OnQueueBatchCreateListener;
import org.autumnframework.service.queue.api.client.listener.batch.OnQueueBatchDeleteListener;
import org.autumnframework.service.queue.api.client.listener.batch.OnQueueBatchUpdateListener;
import org.autumnframework.service.queue.api.messages.IdentifiableMessage;
/**
 * 
 */
@Slf4j
public class CrudMessageListenerAndDelegationService {

    private final ObjectMapper objectMapper;

    //TODO decide to allow multiple implementations per type or not
    private final Map<String, Class<? extends IdentifiableMessage<?>>> classIdToClass;

    private final Map<Class<? extends IdentifiableMessage<?>>, List<CrudQueueListener<?,?>>> classToCrudQueue;

    private final Map<Class<? extends IdentifiableMessage<?>>, List<OnQueueCreateListener>> msgClassToOnCreateListener;
    private final Map<Class<? extends IdentifiableMessage<?>>, List<OnQueueUpdateListener>> msgClassToOnUpdateListener;
    private final Map<Class<? extends IdentifiableMessage<?>>, List<OnQueueDeleteListener>> msgClassToOnDeleteListener;

    /**
     * This service is autowired with a list of all the CrudQueueListeners in the spring context. It listens to all
     * the crud messages and receives the generic AMPQ Message. When a message is received, it uses the header to
     * retrieve the class type of the message, then matches this against the CrudQueueListener class types. If a match
     * is found, objectMapper is used to create the instance of this class and the matched CrudQueueListener is invoked
     * with the specific message. If no match is found, we just trace log and ignore it.
     *
     * To receive messages, all you have to do is implement CrudQueueListener for your specific class type. No queue
     * configuration or whatsoever required! You can register more than one CrudQueueListener
     *
     * You can now also register:
     *  - OnQueueCreateListener
     *  - OnQueueCreateListener
     *  - OnQueueCreateListener
     *
     *  If a message is received, it will be sent to all of the above mentioned queue listeners
     *
     *
     * @param objectMapper
     * @param crudQueueListeners
     * @param onCreateListeners 
     * @param onUpdateListeners 
     * @param onDeleteListeners 
     */
    @Autowired
    public CrudMessageListenerAndDelegationService(ObjectMapper objectMapper,
                                                   List<CrudQueueListener<?, ?>> crudQueueListeners,
                                                   List<OnQueueCreateListener<?, ?>> onCreateListeners,
                                                   List<OnQueueUpdateListener<?, ?>> onUpdateListeners,
                                                   List<OnQueueDeleteListener<?, ?>> onDeleteListeners) {
        this.objectMapper = objectMapper;


        // From all known listeners, collect the message class String to Class types. This map does not only give us insights on what message types exist, it also saves using reflection
        // to try and instantiate a message class that possible does not exist
        this.classIdToClass = crudQueueListeners.stream().collect(Collectors.toMap(crudQueueListener -> crudQueueListener.getMessageType().getName(), CrudQueueListener::getMessageType));

        // Add the ones from create/update/delete listeners
        this.classIdToClass.putAll(onCreateListeners.stream().collect(Collectors.toMap(crudQueueListener -> crudQueueListener.getMessageType().getName(), OnQueueCreateListener::getMessageType)));
        this.classIdToClass.putAll(onUpdateListeners.stream().collect(Collectors.toMap(crudQueueListener -> crudQueueListener.getMessageType().getName(), OnQueueUpdateListener::getMessageType)));
        this.classIdToClass.putAll(onDeleteListeners.stream().collect(Collectors.toMap(crudQueueListener -> crudQueueListener.getMessageType().getName(), OnQueueDeleteListener::getMessageType)));

        this.classToCrudQueue = crudQueueListeners.stream().collect(Collectors.groupingBy(CrudQueueListener::getMessageType));

        this.msgClassToOnCreateListener = onCreateListeners.stream().collect(Collectors.groupingBy(OnQueueCreateListener::getMessageType));
        this.msgClassToOnUpdateListener = onUpdateListeners.stream().collect(Collectors.groupingBy(OnQueueUpdateListener::getMessageType));
        this.msgClassToOnDeleteListener = onDeleteListeners.stream().collect(Collectors.groupingBy(OnQueueDeleteListener::getMessageType));
    }

    @Timed(value = "autumn.messaging.rabbitmq.received", description = "Number of messages received", extraTags = {"routing-key", "in"})
    void handle(Message message,
                BiConsumer<Class<? extends IdentifiableMessage<?>>, IdentifiableMessage> msgClassAndMessage) {
        String classId = message.getMessageProperties().getHeader(DEFAULT_CLASSID_FIELD_NAME);

        String consumerQueue = message.getMessageProperties().getConsumerQueue();
        String messageAction = consumerQueue.substring(consumerQueue.lastIndexOf("."));

        log.trace("Received {} request for {}", messageAction, classId);

        String payload;
        try {
            payload = new String(message.getBody(), message.getMessageProperties().getContentEncoding());
        } catch (UnsupportedEncodingException e) {
            log.error("UnsupportedEncodingExeption while decoding message: {}", e.getMessage());
            throw new RuntimeException("Failed to parse message body to String for classId: " + classId);
        }

        if (!this.classIdToClass.containsKey(classId)) {
            log.trace("Received a message, but no handler was registered in the context that can handle this message. " +
                    "Consider registering a class of OnQueueCreateListener<?, {}>, OnQueueUpdateListener<?, {}>, OnQueueDeleteListener<?, {}> or CrudQueueListener<?, {}> if you need direct access to the IdentifiableMessage instance." +
                    "Message is ignored: {}.", classId, classId, classId, classId, payload);
            return;
        }

        Class<? extends IdentifiableMessage<?>> aClass = this.classIdToClass.get(classId);

        final IdentifiableMessage<?> identifiableMessage;
        try {
            identifiableMessage = this.objectMapper.readValue(payload, aClass);
        } catch (JsonProcessingException e) {
            log.error("JsonProcessingException while mapping message: {}", e.getMessage());
            throw new IllegalStateException("Failed to deserialize message body into class" + aClass, e);
        }

        msgClassAndMessage.accept(aClass, identifiableMessage);

    }
    
    /**
     * @param messages
     * @param msgClassAndMessage
     */
    @Timed(value = "autumn.messaging.rabbitmq.received.bulk", description = "Number of messages received", extraTags = {"routing-key", "in"})
    public void handleList(List<Message> messages,
                BiConsumer<Class<? extends IdentifiableMessage<?>>, List<IdentifiableMessage<?>>> msgClassAndMessage) {
        if (CollectionUtils.isEmpty(messages)) {
            log.warn("handleList called with an empty message list");
            return;
        }
        Message message = messages.get(0);
        String classId = message.getMessageProperties().getHeader(DEFAULT_CLASSID_FIELD_NAME);

        String consumerQueue = message.getMessageProperties().getConsumerQueue();
        String messageAction = consumerQueue.substring(consumerQueue.lastIndexOf("."));

        log.trace("Received batch of {} {} requests for {}", messages.size(), messageAction, classId);

        if (!this.classIdToClass.containsKey(classId)) {
            log.trace("Received a batch of messages, but no handler was registered in the context that can handle this message. " +
                    "Consider registering a class of OnQueueBatchCreateListener<?, {}>, OnQueueBatchUpdateListener<?, {}>, OnQueueBatchDeleteListener<?, {}> or CrudQueueListener<?, {}> if you need direct access to the IdentifiableMessage instance." +
                    "Messages are ignored: {}.", classId, classId, classId, classId, messages);
            return;
        }
        
        Class<? extends IdentifiableMessage<?>> aClass = this.classIdToClass.get(classId);
        final List<IdentifiableMessage<?>> identifiableMessages = new ArrayList<>();

        for (Message current : messages) {
            try {
                identifiableMessages.add(this.objectMapper.readValue(new String(current.getBody(), current.getMessageProperties().getContentEncoding()), aClass));
            } catch (UnsupportedEncodingException e) {
                log.error("UnsupportedEncodingExeption while decoding message: {}", e.getMessage());
                throw new RuntimeException("Failed to parse message body to String for classId: " + classId);
            } catch (JsonProcessingException e) {
                log.error("JsonProcessingException while mapping message: {}", e.getMessage());
                throw new IllegalStateException("Failed to deserialize message body into class" + aClass, e);
            }
        }

        msgClassAndMessage.accept(aClass, identifiableMessages);

    }

    /**
     * @param aClass
     * @param identifiableMessage
     * @param crudQueueListenerConsumerMethod
     */
    public void delegateToCrudQueueListeners(Class<? extends IdentifiableMessage<?>> aClass,
                                              IdentifiableMessage<?> identifiableMessage,
                                              BiConsumer<CrudQueueListener,IdentifiableMessage> crudQueueListenerConsumerMethod) {

        // No generics, but class type is guaranteed because we build the map based on the generic argument class type
        List<CrudQueueListener<?, ?>> crudQueueListeners = this.classToCrudQueue.get(aClass);

        if (CollectionUtils.isEmpty(crudQueueListeners)) {
            log.trace("No CrudQueueListener for message {}", aClass.getSimpleName());
            return;
        }

        crudQueueListeners.forEach(crudQueueListener -> {
            crudQueueListenerConsumerMethod.accept(crudQueueListener, identifiableMessage);
        });
    }

    /**
     * @param aClass
     * @param identifiableMessages
     * @param crudQueueListenerConsumerMethod
     */
    public void delegateListToCrudQueueListeners(Class<? extends IdentifiableMessage<?>> aClass,
                                              List<IdentifiableMessage<?>> identifiableMessages,
                                              BiConsumer<CrudQueueBatchListener,List<IdentifiableMessage<?>>> crudQueueListenerConsumerMethod) {

        // No generics, but class type is guaranteed because we build the map based on the generic argument class type
        List<CrudQueueListener<?, ?>> crudQueueListeners = this.classToCrudQueue.get(aClass);

        if (CollectionUtils.isEmpty(crudQueueListeners)) {
            log.trace("No CrudQueueListener for message {}", aClass.getSimpleName());
            return;
        }

        crudQueueListeners.forEach(crudQueueListener -> {
            crudQueueListenerConsumerMethod.accept((CrudQueueBatchListener)crudQueueListener, identifiableMessages);
        });
    }
    /**
     * @param aClass
     * @param identifiableMessage
     */
    public void delegateToOnCreateListeners( Class<? extends IdentifiableMessage<?>> aClass,
                                              IdentifiableMessage<?> identifiableMessage) {

        List<OnQueueCreateListener> onQueueCreateListeners = this.msgClassToOnCreateListener.get(aClass);

        if (CollectionUtils.isEmpty(onQueueCreateListeners)) {
            log.trace("No OnQueueCreateListener for message {}", aClass.getSimpleName());
            return;
        }

        onQueueCreateListeners.forEach(onCreateListener -> {
            onCreateListener.onCreate(identifiableMessage.getPayload(),
                                        EventMetaData.builder().messageChainId(identifiableMessage.getMessageChainId()).build());
        });
    }

    /**
     * @param aClass
     * @param identifiableMessages
     */
    public void delegateListToOnCreateListeners( Class<? extends IdentifiableMessage<?>> aClass,
                                              List<IdentifiableMessage<?>> identifiableMessages) {

        List<OnQueueCreateListener> onQueueCreateListeners = this.msgClassToOnCreateListener.get(aClass);

        if (CollectionUtils.isEmpty(onQueueCreateListeners)) {
            log.trace("No OnQueueCreateListener for message {}", aClass.getSimpleName());
            return;
        }

        onQueueCreateListeners.forEach(onCreateListener -> {
            if (onCreateListener instanceof OnQueueBatchCreateListener) {
                ((OnQueueBatchCreateListener)onCreateListener).onCreate(
                        identifiableMessages.stream().map(IdentifiableMessage::getPayload).collect(Collectors.toList()), 
                        EventMetaData.builder().messageChainId(identifiableMessages.get(0).getMessageChainId()).build());
            } else {
                for (IdentifiableMessage identifiableMessage : identifiableMessages) {
                    onCreateListener.onCreate(identifiableMessage.getPayload(),
                                                EventMetaData.builder().messageChainId(identifiableMessage.getMessageChainId()).build());
                }
            }
        });
    }

    protected void delegateToOnUpdateListeners( Class<? extends IdentifiableMessage<?>> aClass,
                                              IdentifiableMessage<?> identifiableMessage) {

        List<OnQueueUpdateListener> onQueueUpdateListeners = this.msgClassToOnUpdateListener.get(aClass);
        if (CollectionUtils.isEmpty(onQueueUpdateListeners)) {
            log.trace("No OnQueueUpdateListener for message {}", aClass.getSimpleName());
            return;
        }

        onQueueUpdateListeners.forEach(onUpdateListener -> {
            onUpdateListener.onUpdate(identifiableMessage.getPayload(),
                    EventMetaData.builder().messageChainId(identifiableMessage.getMessageChainId()).build());
        });
    }
    
    /**
     * @param aClass
     * @param identifiableMessages
     */
    public void delegateListToOnUpdateListeners(Class<? extends IdentifiableMessage<?>> aClass,
            List<IdentifiableMessage<?>> identifiableMessages) {

        List<OnQueueUpdateListener> onQueueUpdateListeners = this.msgClassToOnUpdateListener.get(aClass);
        if (CollectionUtils.isEmpty(onQueueUpdateListeners)) {
            log.trace("No OnQueueUpdateListener for message {}", aClass.getSimpleName());
            return;
        }

        onQueueUpdateListeners.forEach(onUpdateListener -> {
            if (onUpdateListener instanceof OnQueueBatchUpdateListener) {
                ((OnQueueBatchUpdateListener)onUpdateListener).onUpdate(
                        identifiableMessages.stream().map(IdentifiableMessage::getPayload).collect(Collectors.toList()), 
                        EventMetaData.builder().messageChainId(identifiableMessages.get(0).getMessageChainId()).build());
            } else {
                for (IdentifiableMessage identifiableMessage : identifiableMessages) {
                    onUpdateListener.onUpdate(identifiableMessage.getPayload(),
                            EventMetaData.builder().messageChainId(identifiableMessage.getMessageChainId()).build());
                }
            }
        });
    }

    protected void delegateToOnDeleteListeners( Class<? extends IdentifiableMessage<?>> aClass,
                                              IdentifiableMessage<?> identifiableMessage) {

        List<OnQueueDeleteListener> onQueueDeleteListeners = this.msgClassToOnDeleteListener.get(aClass);

        if (CollectionUtils.isEmpty(onQueueDeleteListeners)) {
            log.trace("No OnQueueDeleteListener for message {}", aClass.getSimpleName());
            return;
        }

        onQueueDeleteListeners.forEach(onDeleteListener -> {
            onDeleteListener.onDelete(identifiableMessage.getPayload(),
                    EventMetaData.builder().messageChainId(identifiableMessage.getMessageChainId()).build());
        });
    }
    
    /**
     * @param aClass
     * @param identifiableMessages
     */
    public void delegateListToOnDeleteListeners( Class<? extends IdentifiableMessage<?>> aClass,
                                              List<IdentifiableMessage<?>> identifiableMessages) {

        List<OnQueueDeleteListener> onQueueDeleteListeners = this.msgClassToOnDeleteListener.get(aClass);

        if (CollectionUtils.isEmpty(onQueueDeleteListeners)) {
            log.trace("No OnQueueDeleteListener for message {}", aClass.getSimpleName());
            return;
        }

        onQueueDeleteListeners.forEach(onDeleteListener -> {
            if (onDeleteListener instanceof OnQueueBatchDeleteListener) {
                ((OnQueueBatchDeleteListener)onDeleteListener).onDelete(
                        identifiableMessages.stream().map(IdentifiableMessage::getPayload).collect(Collectors.toList()), 
                        EventMetaData.builder().messageChainId(identifiableMessages.get(0).getMessageChainId()).build());
            } else {
                for (IdentifiableMessage identifiableMessage : identifiableMessages) {
                    onDeleteListener.onDelete(identifiableMessage.getPayload(),
                            EventMetaData.builder().messageChainId(identifiableMessage.getMessageChainId()).build());
                }
            }
        });
    }

    /**
     * @param message
     */
    @RabbitListener(queues = {"${autumn.service.name}.in.create"})
    @Timed(value = "autumn.messaging.rabbitmq.received.create", description = "Number of create messages received", extraTags = {"routing-key", "create"})
    void handleCreate(Message message) {
        this.handle(message, (aClass, identifiableMessage) -> {
            this.delegateToCrudQueueListeners(aClass, identifiableMessage, CrudQueueListener::handleCreate);
            this.delegateToOnCreateListeners(aClass, identifiableMessage);
        });
    }

    /**
     * @param message
     */
    @RabbitListener(queues = {"${autumn.service.name}.in.update"})
    @Timed(value = "autumn.messaging.rabbitmq.received.update", description = "Number of update messages received", extraTags = {"routing-key", "update"})
    void handleUpdate(Message message) {
        this.handle(message, (aClass, identifiableMessage) -> {
            this.delegateToCrudQueueListeners(aClass, identifiableMessage, CrudQueueListener::handleUpdate);
            this.delegateToOnUpdateListeners(aClass, identifiableMessage);
        });
    }

    /**
     * @param message
     */
    @RabbitListener(queues = {"${autumn.service.name}.in.delete"})
    @Timed(value = "autumn.messaging.rabbitmq.received.delete", description = "Number of delete messages received", extraTags = {"routing-key", "delete"})
    void handleDelete(Message message) {
        this.handle(message, (aClass, identifiableMessage) -> {
            this.delegateToCrudQueueListeners(aClass, identifiableMessage, CrudQueueListener::handleDelete);
            this.delegateToOnDeleteListeners(aClass, identifiableMessage);
        });
    }

}
