/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.service.rabbit.client.config;

import static org.autumnframework.service.rabbit.api.properties.RabbitMQApiProperties.getAllCreateKey;
import static org.autumnframework.service.rabbit.api.properties.RabbitMQApiProperties.getAllDeleteKey;
import static org.autumnframework.service.rabbit.api.properties.RabbitMQApiProperties.getAllServiceKey;
import static org.autumnframework.service.rabbit.api.properties.RabbitMQApiProperties.getAllUpdateKey;
import static org.autumnframework.service.rabbit.api.properties.RabbitMQApiProperties.getOutExchangeName;
import static org.springframework.amqp.core.BindingBuilder.bind;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.autumnframework.service.queue.api.QueueApiProperties;
import org.springframework.amqp.core.Declarable;
import org.springframework.amqp.core.Declarables;
import org.springframework.amqp.core.Queue;
import org.springframework.amqp.core.TopicExchange;
import org.springframework.amqp.rabbit.annotation.EnableRabbit;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.PropertySource;
import org.springframework.context.annotation.PropertySources;

import com.fasterxml.jackson.databind.ObjectMapper;

import lombok.extern.slf4j.Slf4j;
import org.autumnframework.service.liveness.rabbit.LivenessRabbitConfiguration;
import org.autumnframework.service.queue.api.QueueApiConfiguration;
import org.autumnframework.service.queue.api.client.listener.CrudQueueListener;
import org.autumnframework.service.queue.api.client.listener.OnQueueCreateListener;
import org.autumnframework.service.queue.api.client.listener.OnQueueDeleteListener;
import org.autumnframework.service.queue.api.client.listener.OnQueueUpdateListener;
import org.autumnframework.service.rabbit.api.properties.RabbitMQApiProperties;
import org.autumnframework.service.rabbit.client.listener.CrudMessageListenerAndDelegationService;

/**
 * @author Timon Veenstra
 */
@Configuration
@EnableRabbit
@Import({QueueApiConfiguration.class, MQJsonConfig.class, LivenessRabbitConfiguration.class})
@Slf4j
@PropertySources({@PropertySource("classpath:service.properties")})
public class RabbitClientConfig {

    private static final Map<String, Object> QUEUE_ARGS = new HashMap<>();
    static{
        QUEUE_ARGS.put("x-queue-type", "quorum");
    }


    @Bean
    public CrudMessageListenerAndDelegationService listenerDelegation(ObjectMapper objectMapper,
                                                                      List<CrudQueueListener<?, ?>> crudQueueListeners,
                                                                      List<OnQueueCreateListener<?, ?>> onCreateListeners,
                                                                      List<OnQueueUpdateListener<?, ?>> onUpdateListeners,
                                                                      List<OnQueueDeleteListener<?, ?>> onDeleteListeners) {
        return new CrudMessageListenerAndDelegationService( objectMapper,
                                                            crudQueueListeners,
                                                            onCreateListeners,
                                                            onUpdateListeners,
                                                            onDeleteListeners);
    }

    @Bean
    public Declarables providerBindings(@Autowired QueueApiProperties queueApiProperties,
                                        @Autowired  RabbitMQApiProperties rabbitMQApiProperties) {
        log.trace("creating client bindings");
        List<Declarable> declarables = new ArrayList<>();

        TopicExchange inExchange = new TopicExchange(rabbitMQApiProperties.getInExchangeName());
        declarables.add(inExchange);

        Queue inCreateQueue = new Queue(rabbitMQApiProperties.getInCreateQueue(), true, false, false, QUEUE_ARGS);
        declarables.add(inCreateQueue);
        declarables.add(bind(inCreateQueue).to(inExchange).with(getAllCreateKey()));

        Queue inUpdateQueue = new Queue(rabbitMQApiProperties.getInUpdateQueue(), true, false, false, QUEUE_ARGS);
        declarables.add(inUpdateQueue);
        declarables.add(bind(inUpdateQueue).to(inExchange).with(getAllUpdateKey()));

        Queue inDeleteQueue = new Queue(rabbitMQApiProperties.getInDeleteQueue(), true, false, false, QUEUE_ARGS);
        declarables.add(inDeleteQueue);
        declarables.add(bind(inDeleteQueue).to(inExchange).with(getAllDeleteKey()));

        //
        // link the in exchange to all the services that this listener is subscribed to
        //
        if (queueApiProperties.getSubscribeTo() != null && !queueApiProperties.getSubscribeTo().isEmpty()){
            queueApiProperties.getSubscribeTo().forEach(service -> {
                TopicExchange serviceExchange = new TopicExchange(getOutExchangeName(service));
                declarables.add(serviceExchange);
                declarables.add(bind(inExchange).to(serviceExchange).with(getAllServiceKey(service)));
            });
        }

        return new Declarables(declarables);

    }
}
