/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.service.rabbit.batch.client.listener;

import static org.springframework.amqp.support.converter.AbstractJavaTypeMapper.DEFAULT_CLASSID_FIELD_NAME;

import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

import org.springframework.amqp.core.Message;
import org.springframework.amqp.rabbit.annotation.RabbitListener;

import io.micrometer.core.annotation.Timed;
import org.autumnframework.service.queue.api.client.listener.batch.CrudQueueBatchListener;
import org.autumnframework.service.rabbit.client.listener.CrudMessageListenerAndDelegationService;

public class BatchListenerComponent {
    
    private final CrudMessageListenerAndDelegationService service;
    
    /*
     * This class exists so we can use the ConditionalOnBean on this class instead of on the @RabbitListener methods
     * which for some reason doesn't seem to work. Possibly because they are instantiated differently from @Bean
     */
    public BatchListenerComponent(CrudMessageListenerAndDelegationService service) {
        this.service = service;
    }
    
    /**
     * @param message
     */
    @RabbitListener(queues = {"${autumn.service.name}.in.create"}, containerFactory = "consumerBatchContainerFactory")
    @Timed(value = "autumn.messaging.rabbitmq.received.bulk.create", description = "Number of bulk create messages received", extraTags = {"routing-key", "create"})
    void handleCreate(List<Message> messages) {
        Collection<List<Message>> splitLists = splitList(messages);
        for (List<Message> list : splitLists) {
            this.service.handleList(list, (aClass, identifiableMessages) -> {
                this.service.delegateListToCrudQueueListeners(aClass, identifiableMessages, CrudQueueBatchListener::handleCreateList);
                this.service.delegateListToOnCreateListeners(aClass, identifiableMessages);
            });
        }
    }

    /**
     * @param message
     */
    @RabbitListener(queues = {"${autumn.service.name}.in.update"}, containerFactory = "consumerBatchContainerFactory")
    @Timed(value = "autumn.messaging.rabbitmq.received.bulk.update", description = "Number of bulk update messages received", extraTags = {"routing-key", "update"})
    void handleUpdate(List<Message> messages) {
        Collection<List<Message>> splitLists = splitList(messages);
        for (List<Message> list : splitLists) {
            this.service.handleList(list, (aClass, identifiableMessage) -> {
                this.service.delegateListToCrudQueueListeners(aClass, identifiableMessage, CrudQueueBatchListener::handleUpdateList);
                this.service.delegateListToOnUpdateListeners(aClass, identifiableMessage);
            });
        }
    }

    /**
     * @param message
     */
    @RabbitListener(queues = {"${autumn.service.name}.in.delete"}, containerFactory = "consumerBatchContainerFactory")
    @Timed(value = "autumn.messaging.rabbitmq.received.bulk.delete", description = "Number of bulk delete messages received", extraTags = {"routing-key", "delete"})
    void handleDelete(List<Message> messages) {
        Collection<List<Message>> splitLists = splitList(messages);
        for (List<Message> list : splitLists) {
            this.service.handleList(list, (aClass, identifiableMessages) -> {
                this.service.delegateListToCrudQueueListeners(aClass, identifiableMessages, CrudQueueBatchListener::handleDeleteList);
                this.service.delegateListToOnDeleteListeners(aClass, identifiableMessages);
            });
        }
    }

    private Collection<List<Message>> splitList(List<Message> messages) {
        return messages.stream().collect(Collectors.groupingBy(message -> message.getMessageProperties().getHeader(DEFAULT_CLASSID_FIELD_NAME), Collectors.toList())).values();
    }
}
