/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.service.graphql.scalars.impl;

import static graphql.scalars.util.Kit.typeName;

import java.time.DateTimeException;
import java.time.LocalDateTime;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.util.function.Function;

import graphql.language.StringValue;
import graphql.language.Value;
import graphql.schema.Coercing;
import graphql.schema.CoercingParseLiteralException;
import graphql.schema.CoercingParseValueException;
import graphql.schema.CoercingSerializeException;
import graphql.schema.GraphQLScalarType;

/**
 * 
 */
public class LocalDateTimeScalar {

    /**
     * The singleton ISO-8601 compliant DateTime Scalar
     */
    public static GraphQLScalarType INSTANCE;

    static {
        Coercing<LocalDateTime, String> coercing = new Coercing<>() {
            @Override
            public String serialize(Object input) throws CoercingSerializeException {
                LocalDateTime localDateTime;
                if (input instanceof LocalDateTime) {
                    localDateTime = (LocalDateTime) input;
                } else if (input instanceof ZonedDateTime) {
                    localDateTime = ((ZonedDateTime) input).toLocalDateTime();
                } else if (input instanceof String) {
                    localDateTime = parseLocalDateTime(input.toString(), CoercingSerializeException::new);
                } else {
                    throw new CoercingSerializeException(
                            "Expected something we can convert to 'java.time.LocalDateTime' but was '" + typeName(input) + "'."
                    );
                }
                try {
                    return DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss.SSS").format(localDateTime);
                } catch (DateTimeException e) {
                    throw new CoercingSerializeException(
                            "Unable to turn TemporalAccessor into LocalDateTime because of : '" + e.getMessage() + "'."
                    );
                }
            }

            @Override
            public LocalDateTime parseValue(Object input) throws CoercingParseValueException {
                LocalDateTime localDateTime;
                if (input instanceof LocalDateTime) {
                    localDateTime = (LocalDateTime) input;
                } else if (input instanceof ZonedDateTime) {
                    localDateTime = ((ZonedDateTime) input).toLocalDateTime();
                } else if (input instanceof String) {
                    localDateTime = parseLocalDateTime(input.toString(), CoercingParseValueException::new);
                } else {
                    throw new CoercingParseValueException(
                            "Expected a 'String' but was '" + typeName(input) + "'."
                    );
                }
                return localDateTime;
            }

            @Override
            public LocalDateTime parseLiteral(Object input) throws CoercingParseLiteralException {
                if (!(input instanceof StringValue)) {
                    throw new CoercingParseLiteralException(
                            "Expected AST type 'StringValue' but was '" + typeName(input) + "'."
                    );
                }
                return parseLocalDateTime(((StringValue) input).getValue(), CoercingParseLiteralException::new);
            }

            @Override
            public Value<?> valueToLiteral(Object input) {
                String s = serialize(input);
                return StringValue.newStringValue(s).build();
            }

            private LocalDateTime parseLocalDateTime(String s, Function<String, RuntimeException> exceptionMaker) {
                try {
                    return LocalDateTime.parse(s, DateTimeFormatter.ISO_LOCAL_DATE_TIME);
                } catch (DateTimeParseException e) {
                    throw exceptionMaker.apply("Invalid ISO-8601 value : '" + s + "'. because of : '" + e.getMessage() + "'");
                }
            }
        };

        INSTANCE = GraphQLScalarType.newScalar()
                .name("LocalDateTime")
                .description("An ISO-8601 compliant DateTime Scalar")
                .coercing(coercing)
                .build();
    }

}
