/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.service.graphql.geojson.scalars;

import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.io.ParseException;
import org.locationtech.jts.io.WKTReader;
import org.locationtech.jts.io.WKTWriter;
import org.springframework.context.annotation.Configuration;

import graphql.language.StringValue;
import graphql.language.Value;
import graphql.schema.Coercing;
import graphql.schema.CoercingParseLiteralException;
import graphql.schema.CoercingParseValueException;
import graphql.schema.CoercingSerializeException;
import graphql.schema.GraphQLScalarType;

/**
 *
 * @author perezdf
 */
@Configuration
public class PointScalarConverter {

    public static GraphQLScalarType INSTANCE;

    static {
        Coercing<Point, String> coercing = new Coercing<>() {
            @Override
            public String serialize(Object input) throws CoercingSerializeException {
                if (input instanceof Geometry) {
                    return new WKTWriter().write((Point) input);
                }
                throw new CoercingSerializeException("Expected a Geometry object.");
            }

            @Override
            public Point parseValue(final Object input) {
                try {
                    if (input instanceof String) {
                        return (Point)new WKTReader().read((String) input);
                    }
                    throw new CoercingParseValueException("Expected a String");
                } catch (ParseException ex) {
                    throw new CoercingParseValueException("Error", ex);
                }

            }

            @Override
            public Point parseLiteral(final Object input) {
                if (input instanceof StringValue) {
                    try {
                        return (Point)new WKTReader().read(((StringValue) input).getValue());
                    } catch (ParseException ex) {
                        throw new CoercingParseValueException("Error", ex);
                    }
                }
                throw new CoercingParseLiteralException("Expected a StringValue.");
            }
            
            @Override
            public Value<?> valueToLiteral(Object input) {
                String s = serialize(input);
                return StringValue.newStringValue(s).build();
            }
        };

        INSTANCE = GraphQLScalarType.newScalar()
                .name("Point")
                .description("Point as scalar.")
                .coercing(coercing)
                .build();
    }
}
