/*
 * Copyright © 2016-2023 the original author or authors (info@autumnframework.org)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.autumnframework.service.graphql.dataloader;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.commons.collections4.CollectionUtils;

import graphql.com.google.common.collect.Lists;
import org.autumnframework.service.api.dtos.Identifiable;
import org.autumnframework.service.api.services.elementary.ReadFindByIdsService;
import reactor.core.publisher.Mono;

/**
 * Class to give you a head start on creating a graphql dataloader for the default autumn services and mappers
 */
public class FindByIdsDataLoader {
    /**
     * Basic find by ids method, using a ReadFindByIdsService {@literal <T>} to call findByIds
     * for a set of UUIDs, returning a map of UUID -> T.
     * 
     * @param <T>     The expected return type
     * @param ids     A set of IDs of type UUID
     * @param service The service to use in finding the list of {@literal <T>} (it will call
     *                service.findByIds(ids)
     * @return a Map of {@literal <UUID>}, {@literal <T>} with each {@literal <T>} being the singular return type of
     *         the findByIds method as identified by the UUID key
     */
    public static <T extends Identifiable> Mono<Map<UUID, T>> findByIds(Set<UUID> ids,
            ReadFindByIdsService<T> service) {
        return FindByIdsDataLoader.findByIds(ids, service, (T t) -> t.getId());
    }

    /**
     * Find by ids method, using a ReadFindByIdsService{@literal <T>} to call findByIds for a
     * set of UUIDs, returning a map of UUID -> T. This version uses a specified
     * method to extract the keys (use this to specify a method that differs from
     * T.getId()).
     * 
     * @param <T>          The expected return type
     * @param ids          A set of IDs of type UUID
     * @param keyExtractor A method to extract the UUID key from the received
     *                     objects
     * @param service      The service to use in finding the list of {@literal <T>} (it will
     *                     call service.findByIds(ids)
     * @return a Map of {@literal <UUID>}, {@literal <T>} with each {@literal <T>} being the singular return type of
     *         the findByIds method as identified by the UUID key
     */
    public static <T extends Identifiable> Mono<Map<UUID, T>> findByIds(Set<UUID> ids, ReadFindByIdsService<T> service,
            Function<T, UUID> keyExtractor) {
        return FindByIdsDataLoader.findByKeys(ids, (List<UUID> l) -> service.findByIds(l), keyExtractor,
                (Set<UUID> s) -> new ArrayList<>(s), Function.identity());
    }

    /**
     * Find by ids method, using a specified method taking a List of UUID and
     * returning a Collection of T, returning a map of UUID -> T. This version uses
     * a specified method to retrieve the collection of T.
     * 
     * @param <T>           extends Identifiable
     * @param keys          Set of ids to use in the service method called
     * @param serviceMethod The method to call using the keys received
     * @return a Map of {@literal <R>}, {@literal <T>} with each {@literal <T>} being the singular return type of the
     *         method as identified by key {@literal <R>}
     */
    public static <T extends Identifiable> Mono<Map<UUID, T>> findByIds(Set<UUID> keys,
            Function<List<UUID>, Collection<T>> serviceMethod) {
        return FindByIdsDataLoader.findByKeys(keys, serviceMethod, (T t) -> t.getId(),
                (Set<UUID> s) -> new ArrayList<>(s), Function.identity());
    }

    /**
     * Basic find by keys method, using a specified method taking a List of R and
     * returning a Collection of T, returning a map of R -> T. This version uses a
     * specified method to retrieve the collection of T and a specified method to
     * extract the keys.
     * 
     * @param <T>           extends Identifiable
     * @param <R>           the key type
     * @param keys          Set of ids to use in the service method called
     * @param serviceMethod The method to call using the keys received
     * @param keyExtractor  A method to extract the key from the received objects
     *                      {@literal <R>}
     * @return a Map of {@literal <R>}, {@literal <T>} with each {@literal <T>} being the singular return type of the
     *         method as identified by key {@literal <R>}
     */
    public static <T extends Identifiable, R> Mono<Map<R, T>> findByKeys(Set<R> keys,
            Function<List<R>, Collection<T>> serviceMethod, Function<T, R> keyExtractor) {
        return FindByIdsDataLoader.findByKeys(keys, serviceMethod, keyExtractor, (Set<R> s) -> new ArrayList<>(s),
                Function.identity());
    }

    /**
     * Find by keys method that allows you to specify methods to convert a set and
     * list of keys to the expected parameter type of the service method to use in
     * retrieving the results.
     * 
     * @param <T>                         extends Identifiable
     * @param <C>                         extends Iterable {@literal <R>}
     * @param <R>                         the key type
     * @param keys                        Set of ids to use in the service method
     *                                    called
     * @param serviceMethod               The method to call using the keys received
     * @param keyExtractor                A method to extract the key from the
     *                                    received objects {@literal <R>}
     * @param methodArgumentCreator       A way to convert a Set of {@literal <R>} to the
     *                                    Collection type accepted by serviceMethod
     * @param listToMethodArgumentCreator A way to convert a List of {@literal <R>} to the
     *                                    Collection type accepted by serviceMethod
     * @return a Map of {@literal <R>}, {@literal <T>} with each {@literal <T>} being the singular return type of the
     *         method as identified by key {@literal <R>}
     */
    public static <T extends Identifiable, C extends Iterable<R>, R> Mono<Map<R, T>> findByKeys(Set<R> keys,
            Function<C, Collection<T>> serviceMethod, Function<T, R> keyExtractor,
            Function<Set<R>, C> methodArgumentCreator, Function<List<R>, C> listToMethodArgumentCreator) {
        if (CollectionUtils.isEmpty(keys)) {
            return Mono.empty();
        }
        List<T> result = new ArrayList<>();
        if (keys.size() < 50) {
            return Mono.just(serviceMethod.apply(methodArgumentCreator.apply(keys)).stream()
                    .collect(Collectors.toMap(rel -> keyExtractor.apply(rel), Function.identity())));
        }
        List<List<R>> lists = Lists.partition(new ArrayList<>(keys), 50);
        for (List<R> list : lists) {
            result.addAll(serviceMethod.apply(listToMethodArgumentCreator.apply(list)));
        }
        return Mono
                .just(result.stream().collect(Collectors.toMap(rel -> keyExtractor.apply(rel), Function.identity())));
    }
}
