/*
 * Copyright (c) 2008 appspy.org, Contributors.
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0 
 * Unless required by applicable law or agreed to in writing, software distributed under the 
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language governing permissions 
 * and limitations under the License. */

package org.appspy.perf.servlet;

import java.io.File;
import java.io.FileFilter;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Properties;

import javax.servlet.ServletContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.appspy.client.common.Collector;
import org.appspy.client.common.CollectorFactory;
import org.appspy.client.common.CollectorInfo;
import org.appspy.client.common.config.ConfigLoader;

/**
 * @author Olivier HEDIN / olivier@appspy.org
 */
public class AppSpyWebappManagerFactory {
	
	public static final String APPSPY_WEBAPP_MANAGER = "org.appspy.webapp.manager";
	
	public static final String WEBAPP_CONFIG_LOADER_PARAM_NAME = "org.appspy.webappConfigLoader";
	public static final String DEFAULT_WEBAPP_CONFIG_LOADER = org.appspy.client.common.config.FileConfigLoader.class.getName();
	public static final String WEBAPP_CONFIG_PATTERN_PARAM_NAME = "org.appspy.webappConfigPattern";
	public static final String DEFAULT_WEBAPP_CONFIG_PATTERN = "appspy-webapp-{0}.xml";

	protected static Log sLog = LogFactory.getLog(AppSpyWebappManagerFactory.class);

	protected static Integer sLock = new Integer(0);
	
	public static void initAppSpyWebappManager(ServletContext servletContext) throws Exception
	{
		AppSpyWebappManager result = null;
		
		result = (AppSpyWebappManager) servletContext.getAttribute(APPSPY_WEBAPP_MANAGER);
		
		if (result == null)
		{
			synchronized(sLock)
			{
				result = (AppSpyWebappManager) servletContext.getAttribute(APPSPY_WEBAPP_MANAGER);
				
				if (result == null)
				{
					sLog.info("Creation of the AppSpyWebappManager");
					
					if (sLog.isInfoEnabled())
					{
						sLog.info("Using appspy dir : " + CollectorFactory.getAppspyDir());
					}
					
					Properties configProperties = getConfigProperties(servletContext);
					
					initCollector(servletContext, configProperties);
					
					File appSpyLibDir = new File(CollectorFactory.getAppspyDir(), "client/servlet/lib");
					
					File[] libs = appSpyLibDir.listFiles(new FileFilter(){

						public boolean accept(File pathname) {
							if (pathname.isFile() && pathname.getName().endsWith(".jar"))
							{
								return true;
							}
							return false;
						}});
					
					ArrayList<URL> urls = new ArrayList<URL>();
					
					for (File file : libs) {
						try {
							urls.add(file.toURL());
						} catch (MalformedURLException e) {
							e.printStackTrace();
						}
					}
					
					if (sLog.isInfoEnabled())
					{
						sLog.info("Libraries used in the webappManager classloader : " + urls);
					}
					
					String webappConfigLoaderClassName = DEFAULT_WEBAPP_CONFIG_LOADER;
					if (servletContext.getInitParameter(WEBAPP_CONFIG_LOADER_PARAM_NAME) != null)
					{
						webappConfigLoaderClassName = servletContext.getInitParameter(WEBAPP_CONFIG_LOADER_PARAM_NAME);
					}
					ConfigLoader webappConfigLoader = (ConfigLoader) Class.forName(webappConfigLoaderClassName, true, Thread.currentThread().getContextClassLoader()).newInstance();
					
					String webappConfigPattern = DEFAULT_WEBAPP_CONFIG_PATTERN;
					if (servletContext.getInitParameter(WEBAPP_CONFIG_PATTERN_PARAM_NAME) != null)
					{
						webappConfigPattern = servletContext.getInitParameter(WEBAPP_CONFIG_PATTERN_PARAM_NAME);
					}

					String webappConfigLocation = MessageFormat.format(webappConfigPattern, CollectorFactory.getAppspyMode());
					URL webappConfigURL = webappConfigLoader.getURL(webappConfigLocation);

					URLClassLoader cl = new URLClassLoader(urls.toArray(new URL[]{}), Thread.currentThread().getContextClassLoader());
					try {
						Class<?> managerClass = Class.forName("org.appspy.perf.servlet.internal.AppSpyWebappManagerImpl", true, cl);
						result = (AppSpyWebappManager) managerClass.newInstance();
						result.init(configProperties, webappConfigURL);
						servletContext.setAttribute(APPSPY_WEBAPP_MANAGER, result);
					} catch (Exception e) {
						sLog.error("Error in webappManager initialization", e);
						CollectorFactory.getCollectorInfo().setStatus(CollectorInfo.STATUS_BAD_CONFIG, e);
					}
				}
			}
		}
	}
	
	protected static Collector initCollector(ServletContext servletContext, Properties configProperties) throws Exception
	{
		String collectorConfigLoaderClassName = CollectorFactory.DEFAULT_COLLECTOR_CONFIG_LOADER;
		if (servletContext.getInitParameter(CollectorFactory.COLLECTOR_CONFIG_LOADER_PARAM_NAME) != null)
		{
			collectorConfigLoaderClassName = servletContext.getInitParameter(CollectorFactory.COLLECTOR_CONFIG_LOADER_PARAM_NAME);
		}
		ConfigLoader collectorConfigLoader = (ConfigLoader) Class.forName(collectorConfigLoaderClassName, true, Thread.currentThread().getContextClassLoader()).newInstance();
		
		String collectorConfigPattern = CollectorFactory.DEFAULT_COLLECTOR_CONFIG_PATTERN;
		if (servletContext.getInitParameter(CollectorFactory.COLLECTOR_CONFIG_PATTERN_PARAM_NAME) != null)
		{
			collectorConfigPattern = servletContext.getInitParameter(CollectorFactory.COLLECTOR_CONFIG_PATTERN_PARAM_NAME);
		}

		String collectorConfigLocation = MessageFormat.format(collectorConfigPattern, CollectorFactory.getAppspyMode());
		URL collectorConfigURL = collectorConfigLoader.getURL(collectorConfigLocation);
		
		Collector collector = CollectorFactory.initCollector(configProperties, collectorConfigURL);
		
		return collector;
	}
	
	public static AppSpyWebappManager getAppSpyWebappManager(ServletContext servletContext)
	{
		AppSpyWebappManager result = null;
		
		result = (AppSpyWebappManager) servletContext.getAttribute(APPSPY_WEBAPP_MANAGER);
		
		return result;
	}
	
	protected static Properties getConfigProperties(ServletContext servletContext) throws Exception
	{
		String propertiesConfigLoaderClassName = CollectorFactory.DEFAULT_PROPERTIES_CONFIG_LOADER;
		if (servletContext.getInitParameter(CollectorFactory.PROPERTIES_CONFIG_LOADER_PARAM_NAME) != null)
		{
			propertiesConfigLoaderClassName = servletContext.getInitParameter(CollectorFactory.PROPERTIES_CONFIG_LOADER_PARAM_NAME);
		}
		ConfigLoader propertiesConfigLoader = (ConfigLoader) Class.forName(propertiesConfigLoaderClassName, true, Thread.currentThread().getContextClassLoader()).newInstance();
		
		String propertiesConfigPattern = CollectorFactory.DEFAULT_PROPERTIES_CONFIG_PATTERN;
		if (servletContext.getInitParameter(CollectorFactory.PROPERTIES_CONFIG_PATTERN_PARAM_NAME) != null)
		{
			propertiesConfigPattern = servletContext.getInitParameter(CollectorFactory.PROPERTIES_CONFIG_PATTERN_PARAM_NAME);
		}
		
		String propertiesConfigLocation = MessageFormat.format(propertiesConfigPattern, CollectorFactory.getAppspyMode());
		Properties properties = propertiesConfigLoader.loadConfig(propertiesConfigLocation);
		
		return properties;
	}
}
