/*
 * Copyright (c) 2008 appspy.org, Contributors.
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0 
 * Unless required by applicable law or agreed to in writing, software distributed under the 
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language governing permissions 
 * and limitations under the License. */

package org.appspy.perf.servlet;

import java.io.IOException;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.appspy.client.common.Collector;
import org.appspy.client.common.CollectorFactory;
import org.appspy.client.common.CollectorInfo;
import org.appspy.perf.data.PerfTimerType;
import org.appspy.perf.data.ServletTimerData;
import org.appspy.perf.servlet.provider.DataProvider;
import org.appspy.perf.servlet.skip.SkipManager;

/**
 * @author Olivier HEDIN / olivier@appspy.org
 */
public class AppSpyFilter implements Filter {
	
	protected Log sLog = LogFactory.getLog(AppSpyFilter.class);
	protected FilterConfig mFilterConfig = null;

	/* (non-Javadoc)
	 * @see javax.servlet.Filter#destroy()
	 */
	public void destroy() {
		AppSpyWebappManagerFactory.getAppSpyWebappManager(mFilterConfig.getServletContext()).destroy();
	}

	/* (non-Javadoc)
	 * @see javax.servlet.Filter#doFilter(javax.servlet.ServletRequest, javax.servlet.ServletResponse, javax.servlet.FilterChain)
	 */
	public void doFilter(ServletRequest req, ServletResponse res, FilterChain chain) throws IOException,
			ServletException {
		
		sLog.debug("Entering into AppSpyFilter");
		CollectorInfo collectorInfo = CollectorFactory.getCollectorInfo();

		if (sLog.isDebugEnabled())
		{
			sLog.debug("AppSpy Collector Status is " + collectorInfo.getStatus());
		}
		
		boolean skipRequest = false;
		
		if (collectorInfo.getStatus() == CollectorInfo.STATUS_BAD_CONFIG || collectorInfo.getStatus() == CollectorInfo.STATUS_STOPPED)
		{
			sLog.debug("AppSpyFilter will skip this request (cause : status)");
			skipRequest = true;
		}
		else
		{
			SkipManager skipManager = AppSpyWebappManagerFactory.getAppSpyWebappManager(mFilterConfig.getServletContext()).getSkipManager();
			int skipRequestResult = skipManager.skipRequest(req, res, mFilterConfig.getServletContext()); 
			skipRequest = (skipRequestResult == SkipManager.SKIP);
		}
		
		if (skipRequest)
		{
			chain.doFilter(req, res);
		}
		else
		{
			sLog.debug("AppSpyFilter starts the processing of request");
			HttpServletRequest request = (HttpServletRequest) req;
			HttpServletResponse response = (HttpServletResponse) res;
			
			AppSpyServletRequestWrapper requestWrapper = new AppSpyServletRequestWrapper(request);
			AppSpyServletResponseWrapper responseWrapper = new AppSpyServletResponseWrapper(response);
			
			Throwable resultThrowable = null;
			
			DataProvider servletTimerDataProvider = null;
			Collector collector = null;
			ServletTimerData servletTimerData = new ServletTimerData();
			
			try
			{
				collector = CollectorFactory.getCollector();
				sLog.debug("Starting data collection");
				collector.beginCollection(servletTimerData);
				servletTimerDataProvider = (DataProvider) AppSpyWebappManagerFactory.getAppSpyWebappManager(mFilterConfig.getServletContext()).getWebappDataProvider();
				if (sLog.isDebugEnabled())
				{
					sLog.debug("Using this dataProvider : " + servletTimerDataProvider);
				}
				sLog.debug("servletTimerDataProvider.beforeRequest");
				servletTimerDataProvider.beforeRequest(servletTimerData, requestWrapper, responseWrapper, mFilterConfig.getServletContext());
			}
			catch (Throwable t)
			{
				sLog.error("Error while processing the request", t);
				if (collectorInfo != null)
				{
					collectorInfo.setStatus(CollectorInfo.STATUS_ERROR, t);
				}
			}
			
			try
			{
				sLog.debug("Before chain.doFilter");
				chain.doFilter(requestWrapper, responseWrapper);
				sLog.debug("After return of chain.doFilter");
			}
			catch (Throwable t)
			{
				resultThrowable = t;
				sLog.debug("After throwable of chain.doFilter");
			}
			finally
			{
				try
				{
					servletTimerData.setTimerType(PerfTimerType.SERVER);
					sLog.debug("servletTimerDataProvider.afterRequest");
					servletTimerDataProvider.afterRequest(servletTimerData, requestWrapper, responseWrapper, mFilterConfig.getServletContext(), resultThrowable);
					sLog.debug("End of data collection");
					collector.endCollection();
				}
				catch (Throwable t)
				{
					sLog.error("Error while collecting data", t);
					if (collectorInfo != null)
					{
						collectorInfo.setStatus(CollectorInfo.STATUS_ERROR, t);
					}
				}
			}
			
			if (resultThrowable != null)
			{
				if (resultThrowable instanceof IOException)
				{
					throw (IOException) resultThrowable;
				}
				else if (resultThrowable instanceof ServletException)
				{
					throw (ServletException) resultThrowable;
				}
				else if (resultThrowable instanceof RuntimeException)
				{
					throw (RuntimeException) resultThrowable;
				}
				else
				{
					throw new RuntimeException(resultThrowable);
				}
			}
		}
		
	}

	/* (non-Javadoc)
	 * @see javax.servlet.Filter#init(javax.servlet.FilterConfig)
	 */
	public void init(FilterConfig config) throws ServletException {
		mFilterConfig = config;
		
		ClassLoader cl = Thread.currentThread().getContextClassLoader();
		try
		{
			Thread.currentThread().setContextClassLoader(Collector.class.getClassLoader());
			AppSpyWebappManagerFactory.initAppSpyWebappManager(config.getServletContext());
		}
		catch (Throwable t)
		{
			sLog.error("Error in init of AppSpyFilter", t);
		}
		finally
		{
			Thread.currentThread().setContextClassLoader(cl);
		}
	}

}
