/*********************************************************************************
 *                                                                               *
 * The MIT License (MIT)                                                         *
 *                                                                               *
 * Copyright (c) 2015-2022 aoju.org OSHI and other contributors.                 *
 *                                                                               *
 * Permission is hereby granted, free of charge, to any person obtaining a copy  *
 * of this software and associated documentation files (the "Software"), to deal *
 * in the Software without restriction, including without limitation the rights  *
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell     *
 * copies of the Software, and to permit persons to whom the Software is         *
 * furnished to do so, subject to the following conditions:                      *
 *                                                                               *
 * The above copyright notice and this permission notice shall be included in    *
 * all copies or substantial portions of the Software.                           *
 *                                                                               *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR    *
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,      *
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE   *
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER        *
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, *
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN     *
 * THE SOFTWARE.                                                                 *
 *                                                                               *
 ********************************************************************************/
package org.aoju.bus.health;

import org.aoju.bus.core.annotation.NotThreadSafe;
import org.aoju.bus.core.lang.Normal;
import org.aoju.bus.core.lang.Symbol;

import java.util.Properties;

/**
 * The global configuration utility
 *
 * @author Kimi Liu
 * @since Java 17+
 */
@NotThreadSafe
public final class Config {

    public static final String PROPERTIES = "bus.health.properties";
    public static final String FILENAME_PROPERTIES = "bus.health.linux.filename.properties";
    public static final String MACOS_VERSIONS_PROPERTIES = "bus.health.macos.version.properties";
    public static final String VM_MAC_ADDR_PROPERTIES = "bus.health.vmmacaddr.properties";
    public static final String ARCHITECTURE_PROPERTIES = "bus.health.architecture.properties";
    public static final String MEMOIZER_EXPIRATION = "bus.health.memoizer.expiration";
    public static final String WMI_TIMEOUT = "bus.health.wmi.timeout";
    public static final String PROC_PATH = "bus.health.proc.path";
    public static final String PSEUDO_FILESYSTEM_TYPES = "bus.health.pseudo.filesystem.types";
    public static final String NETWORK_FILESYSTEM_TYPES = "bus.health.network.filesystem.types";
    public static final String OS_WINDOWS_EVENTLOG = "bus.health.os.windows.eventlog";
    public static final String OS_WINDOWS_PROCSTATE_SUSPENDED = "bus.health.os.windows.procstate.suspended";
    public static final String OS_WINDOWS_COMMANDLINE_BATCH = "bus.health.os.windows.commandline.batch";
    public static final String OS_WINDOWS_HKEYPERFDATA = "bus.health.os.windows.hkeyperfdata";
    public static final String OSHI_OS_WINDOWS_LOADAVERAGE = "bus.health.os.windows.loadaverage";
    public static final String OS_WINDOWS_CPU_UTILITY = "bus.health.os.windows.cpu.utility";
    public static final String OS_WINDOWS_PERFDISK_DIABLED = "bus.os.windows.perfdisk.disabled";
    public static final String OS_WINDOWS_PERFOS_DIABLED = "bus.os.windows.perfos.disabled";
    public static final String OS_WINDOWS_PERFPROC_DIABLED = "bus.os.windows.perfproc.disabled";
    public static final String OS_UNIX_WHOCOMMAND = "bus.health.os.unix.whoCommand";
    private static final Properties CONFIG = readProperties(PROPERTIES);

    /**
     * Get the property associated with the given key.
     *
     * @param key The property key
     * @return The property value if it exists, or null otherwise
     */
    public static String get(String key) {
        return CONFIG.getProperty(key);
    }

    /**
     * Get the {@code String} property associated with the given key.
     *
     * @param key The property key
     * @param def The default value
     * @return The property value or the given default if not found
     */
    public static String get(String key, String def) {
        return CONFIG.getProperty(key, def);
    }

    /**
     * Get the {@code int} property associated with the given key.
     *
     * @param key The property key
     * @param def The default value
     * @return The property value or the given default if not found
     */
    public static int get(String key, int def) {
        String value = CONFIG.getProperty(key);
        return null == value ? def : Builder.parseIntOrDefault(value, def);
    }

    /**
     * Get the {@code double} property associated with the given key.
     *
     * @param key The property key
     * @param def The default value
     * @return The property value or the given default if not found
     */
    public static double get(String key, double def) {
        String value = CONFIG.getProperty(key);
        return null == value ? def : Builder.parseDoubleOrDefault(value, def);
    }

    /**
     * Get the {@code boolean} property associated with the given key.
     *
     * @param key The property key
     * @param def The default value
     * @return The property value or the given default if not found
     */
    public static boolean get(String key, boolean def) {
        String value = CONFIG.getProperty(key);
        return null == value ? def : Boolean.parseBoolean(value);
    }

    /**
     * Set the given property, overwriting any existing value. If the given value is
     * {@code null}, the property is removed.
     *
     * @param key The property key
     * @param val The new value
     */
    public static void set(String key, Object val) {
        if (null == val) {
            CONFIG.remove(key);
        } else {
            CONFIG.setProperty(key, val.toString());
        }
    }

    /**
     * Reset the given property to its default value.
     *
     * @param key The property key
     */
    public static void remove(String key) {
        CONFIG.remove(key);
    }

    /**
     * Clear the configuration.
     */
    public static void clear() {
        CONFIG.clear();
    }

    /**
     * Load the given {@link java.util.Properties} into the global configuration.
     *
     * @param properties The new properties
     */
    public static void load(Properties properties) {
        CONFIG.putAll(properties);
    }

    /**
     * Read a configuration file from the class path and return its properties
     *
     * @param fileName The filename
     * @return A {@link java.util.Properties} object containing the properties.
     */
    public static Properties readProperties(String fileName) {
        return org.aoju.bus.setting.magic.Properties.getProp(Symbol.SLASH + Normal.META_DATA_INF + "/health/" + fileName, Builder.class);
    }

    /**
     * Indicates that a configuration value is invalid.
     */
    public static class PropertyException extends RuntimeException {

        private static final long serialVersionUID = -7482581936621748005L;

        /**
         * @param property The property name
         */
        public PropertyException(String property) {
            super("Invalid property: \"" + property + "\" = " + Config.get(property, null));
        }

        /**
         * @param property The property name
         * @param message  An exception message
         */
        public PropertyException(String property, String message) {
            super("Invalid property \"" + property + "\": " + message);
        }
    }

}
