/*
 * Decompiled with CFR 0.152.
 */
package alluxio.cli.hdfs;

import alluxio.cli.AbstractValidationTask;
import alluxio.cli.ApplicableUfsType;
import alluxio.cli.ValidationTaskResult;
import alluxio.cli.ValidationUtils;
import alluxio.cli.hdfs.HadoopConfigurationFileParser;
import alluxio.collections.Pair;
import alluxio.conf.AlluxioConfiguration;
import alluxio.conf.PropertyKey;
import alluxio.exception.InvalidPathException;
import alluxio.util.ExceptionUtils;
import alluxio.util.io.PathUtils;
import java.io.IOException;
import java.net.URI;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import javax.annotation.Nullable;

@ApplicableUfsType(value=ApplicableUfsType.Type.HDFS)
public class HdfsConfValidationTask
extends AbstractValidationTask {
    public static final String SEPARATOR = ":";
    protected final AlluxioConfiguration mConf;
    final String mPath;
    Map<String, String> mCoreConf = null;
    Map<String, String> mHdfsConf = null;
    ValidationUtils.State mState = ValidationUtils.State.OK;
    StringBuilder mMsg = new StringBuilder();
    StringBuilder mAdvice = new StringBuilder();

    public HdfsConfValidationTask(String path, AlluxioConfiguration conf) {
        this.mPath = path;
        this.mConf = conf;
    }

    public String getName() {
        return "ValidateHdfsConf";
    }

    protected ValidationTaskResult loadHdfsConfig() {
        Pair<String, String> clientConfFiles = this.getHdfsConfPaths();
        String coreConfPath = (String)clientConfFiles.getFirst();
        String hdfsConfPath = (String)clientConfFiles.getSecond();
        this.mCoreConf = this.accessAndParseConf("core-site.xml", coreConfPath);
        this.mHdfsConf = this.accessAndParseConf("hdfs-site.xml", hdfsConfPath);
        return new ValidationTaskResult(this.mState, this.getName(), this.mMsg.toString(), this.mAdvice.toString());
    }

    protected Pair<String, String> getHdfsConfPaths() {
        String confVal = this.mConf.getString(PropertyKey.UNDERFS_HDFS_CONFIGURATION);
        String[] clientHadoopConfFilePaths = confVal.split(SEPARATOR);
        this.mMsg.append(String.format("%d file path(s) detected in for HDFS configuration files for \"%s\"%n", clientHadoopConfFilePaths.length, confVal));
        String clientCoreSiteFilePath = null;
        String clientHdfsSiteFilePath = null;
        for (String path : clientHadoopConfFilePaths) {
            if (path.contains("core-site.xml")) {
                clientCoreSiteFilePath = path;
                continue;
            }
            if (!path.contains("hdfs-site.xml")) continue;
            clientHdfsSiteFilePath = path;
        }
        if (clientHadoopConfFilePaths.length < 2 || clientCoreSiteFilePath == null || clientHdfsSiteFilePath == null) {
            this.mAdvice.append(String.format("Additional HDFS configuration can be specified with your Alluxio mount point by configuring the property %s. The value for this property should be in the format \"{core-site.xml path}:{hdfs-site.xml path}\"%n", PropertyKey.UNDERFS_HDFS_CONFIGURATION.getName()));
        }
        return new Pair((Object)clientCoreSiteFilePath, clientHdfsSiteFilePath);
    }

    public ValidationTaskResult validateImpl(Map<String, String> optionsMap) {
        if (!ValidationUtils.isHdfsScheme((String)this.mPath)) {
            this.mMsg.append(String.format("UFS path %s is not HDFS. Skipping validation for HDFS properties.%n", this.mPath));
            return new ValidationTaskResult(ValidationUtils.State.SKIPPED, this.getName(), this.mMsg.toString(), this.mAdvice.toString());
        }
        ValidationTaskResult loadConfig = this.loadHdfsConfig();
        if (loadConfig.getState() != ValidationUtils.State.OK) {
            return loadConfig;
        }
        ValidationTaskResult last = this.checkConflicts();
        if (last.getState() == ValidationUtils.State.OK) {
            last = this.checkNameservice();
        }
        return last;
    }

    protected ValidationTaskResult checkNameservice() {
        ValidationUtils.State state;
        String nsKey = "dfs.nameservices";
        String nameservices = this.mCoreConf.get(nsKey);
        if (nameservices == null) {
            nameservices = this.mHdfsConf.get(nsKey);
        }
        if (nameservices == null) {
            return new ValidationTaskResult(ValidationUtils.State.OK, this.getName(), "No nameservice detected", "");
        }
        List nsList = Arrays.stream(nameservices.split(",")).map(String::trim).collect(Collectors.toList());
        try {
            URI u = URI.create(this.mPath);
            String uriHost = u.getHost().toLowerCase();
            long nsCount = nsList.stream().filter(s -> uriHost.equals(s.toLowerCase())).count();
            state = ValidationUtils.State.OK;
            if (nsCount < 1L) {
                state = ValidationUtils.State.FAILED;
                this.mAdvice.append(String.format("One or more nameservices (%s) were detected in the HDFS configuration, but not used in the URI to connect to HDFS.", nameservices));
                this.mMsg.append(String.format("Could not find any of the configured nameservices (%s) in the given HDFS connection URI (%s)", nameservices, u));
            }
        }
        catch (IllegalArgumentException e) {
            state = ValidationUtils.State.FAILED;
            this.mMsg.append("HDFS path not parsable as a URI.");
            this.mMsg.append(ExceptionUtils.asPlainText((Throwable)e));
            this.mAdvice.append("Make sure the HDFS URI is in a valid format.");
        }
        return new ValidationTaskResult(state, this.getName(), this.mMsg.toString(), this.mAdvice.toString());
    }

    protected ValidationTaskResult checkConflicts() {
        ValidationUtils.State state = ValidationUtils.State.OK;
        for (Map.Entry<String, String> entry : this.mCoreConf.entrySet()) {
            String coreValue;
            String hdfsValue;
            String k = entry.getKey();
            if (!this.mHdfsConf.containsKey(k) || (hdfsValue = this.mHdfsConf.get(k)).equals(coreValue = this.mCoreConf.get(k))) continue;
            state = ValidationUtils.State.FAILED;
            this.mMsg.append(String.format("Property %s is %s in core-site.xml and %s in hdfs-site.xml", k, coreValue, hdfsValue));
            this.mAdvice.append(String.format("Please fix the inconsistency for %s in core-site.xml and hdfs.xml.%n", k));
        }
        if (state == ValidationUtils.State.OK) {
            this.mMsg.append("core-site.xml and hdfs-site.xml are consistent.\n");
        }
        return new ValidationTaskResult(state, this.getName(), this.mMsg.toString(), this.mAdvice.toString());
    }

    @Nullable
    private Map<String, String> accessAndParseConf(String configName, String path) {
        if (path == null || path.isEmpty()) {
            this.mMsg.append(String.format("%s is not configured in Alluxio property %s%n", configName, PropertyKey.UNDERFS_HDFS_CONFIGURATION));
            this.mState = ValidationUtils.State.SKIPPED;
            return null;
        }
        try {
            PathUtils.getPathComponents((String)path);
        }
        catch (InvalidPathException e) {
            this.mState = ValidationUtils.State.WARNING;
            this.mMsg.append(String.format("Invalid path %s in Alluxio property %s.%n", path, PropertyKey.UNDERFS_HDFS_CONFIGURATION));
            this.mMsg.append(ExceptionUtils.asPlainText((Throwable)e));
            this.mAdvice.append(String.format("Please correct the path for %s in %s%n", configName, PropertyKey.UNDERFS_HDFS_CONFIGURATION));
            return null;
        }
        Path confPath = Paths.get(path, new String[0]);
        if (!Files.exists(confPath, new LinkOption[0])) {
            this.mState = ValidationUtils.State.WARNING;
            this.mMsg.append(String.format("File does not exist at %s in Alluxio property %s.%n", path, PropertyKey.UNDERFS_HDFS_CONFIGURATION));
            this.mAdvice.append(String.format("Could not file file at \"%s\". Correct the path in %s%n", configName, PropertyKey.UNDERFS_HDFS_CONFIGURATION));
            return null;
        }
        if (!Files.isReadable(Paths.get(path, new String[0]))) {
            this.mState = ValidationUtils.State.WARNING;
            this.mMsg.append(String.format("\"%s\" is not readable from Alluxio property %s.%n", path, PropertyKey.UNDERFS_HDFS_CONFIGURATION));
            this.mAdvice.append(String.format("Grant more accessible permissions on the file %s from %s%n", configName, PropertyKey.UNDERFS_HDFS_CONFIGURATION));
            return null;
        }
        HadoopConfigurationFileParser parser = new HadoopConfigurationFileParser();
        Map<String, String> properties = null;
        try {
            properties = parser.parseXmlConfiguration(path);
            this.mMsg.append(String.format("Successfully loaded %s. %n", path));
        }
        catch (IOException e) {
            this.mState = ValidationUtils.State.FAILED;
            this.mMsg.append(String.format("Failed to read %s. %s.%n", path, e.getMessage()));
            this.mMsg.append(ExceptionUtils.asPlainText((Throwable)e));
            this.mAdvice.append(String.format("Please check your %s.%n", path));
        }
        catch (RuntimeException e) {
            this.mState = ValidationUtils.State.FAILED;
            this.mMsg.append(String.format("Failed to parse %s. %s.%n", path, e.getMessage()));
            this.mMsg.append(ExceptionUtils.asPlainText((Throwable)e));
            this.mAdvice.append(String.format("Failed to parse %s as valid XML. Please check that the file path is correct and the content is valid XML.%n", path));
        }
        return properties;
    }
}

