/*
 * MentaQueue => http://mentaqueue.soliveirajr.com
 * Copyright (C) 2012  Sergio Oliveira Jr. (sergio.oliveira.jr@gmail.com)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package org.mentaqueue.test.producer;

import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;

import org.mentaqueue.AtomicQueue;
import org.mentaqueue.BlockingArrayQueue;
import org.mentaqueue.BlockingLinkedQueue;
import org.mentaqueue.BlockingQueue;
import org.mentaqueue.BrokenFastQueue;
import org.mentaqueue.ConcurrentLinkedQueue;
import org.mentaqueue.LockedQueue;
import org.mentaqueue.SynchronizedQueue;
import org.mentaqueue.VolatileQueue;
import org.mentaqueue.pooled.PooledBlockingArrayQueue;
import org.mentaqueue.pooled.PooledBlockingLinkedQueue;
import org.mentaqueue.pooled.PooledBlockingQueue;
import org.mentaqueue.pooled.PooledConcurrentLinkedQueue;
/**
 * <p>
 * To run: 
 * <pre>
 * java -Xms1g -Xmx4g -XX:NewSize=512m -XX:MaxNewSize=1024m -DtestNonBatchingQueues=true -cp target/classes org.mentaqueue.test.producer.TestAll 20 5 50 1000 1024
 * 
 * format: java -Xms1g -Xmx4g -XX:NewSize=512m -XX:MaxNewSize=1024m -DtestUncheckedQueues=true -DtestNonBatchingQueues=true -cp target/classes org.mentaqueue.test.producer.TestAll IGNORE OUTLIERS RUNS MESSAGES BUFFER_SIZE
 * IGNORE => How many initial runs will be ignored when calculating the average (warmup runs)?
 * OUTLIERS => How many best and worst to remove?
 * RUNS => How many times to run the test?
 * MESSAGES => How many messages to send to the other thread? (in thousands)
 * BUFFER_SIZE => How many messages can the buffer hold? (in multiples of 1024)
 * </pre>
 * </p>
 * <p>
 * Latest results:
 * <pre>
 * ******************************************************************
 *   LINUX
 * ******************************************************************
 * 
 * Intel(R) Xeon(R) CPU E5345 @ 2.33GHz
 * One physical processor with 6 logical processors
 * Java HotSpot(TM) 64-Bit Server VM (build 20.2-b06, mixed mode)
 * Linux soliveira-linux 2.6.32-30-generic x86_64 GNU/Linux
 * 
 * Final results:
 * 
 * 1: BrokenFastQueue => 0 nanoseconds/message
 * 2: <b>AtomicQueue => 6 nanoseconds/message</b>
 * 3: VolatileQueue => 10 nanoseconds/message (1.67x)
 * 4: SynchronizedQueue => 20 nanoseconds/message (3.33x)
 * 5: LockedQueue => 41 nanoseconds/message (6.83x)
 * 6: AtomicQueue (*) => 49 nanoseconds/message (8.17x)
 * 7: ConcurrentLinkedQueue => 68 nanoseconds/message (11.33x)
 * 8: VolatileQueue (*) => 73 nanoseconds/message (12.17x)
 * 9: PooledConcurrentLinkedQueue => 134 nanoseconds/message (22.33x)
 * 10: PooledBlockingQueue => 172 nanoseconds/message (28.67x)
 * 11: BlockingQueue => 179 nanoseconds/message (29.83x)
 * 12: SynchronizedQueue (*) => 185 nanoseconds/message (30.83x)
 * 13: PooledBlockingLinkedQueue => 248 nanoseconds/message (41.33x)
 * 14: BlockingLinkedQueue => 249 nanoseconds/message (41.5x)
 * 15: BlockingLinkedQueue (*) => 265 nanoseconds/message (44.17x)
 * 16: BlockingQueue (*) => 267 nanoseconds/message (44.5x)
 * 17: BlockingArrayQueue => 278 nanoseconds/message (46.33x)
 * 18: PooledBlockingLinkedQueue (*) => 279 nanoseconds/message (46.5x)
 * 19: LockedQueue (*) => 316 nanoseconds/message (52.67x)
 * 20: PooledBlockingQueue (*) => 321 nanoseconds/message (53.5x)
 * 21: PooledBlockingArrayQueue => 323 nanoseconds/message (53.83x)
 * 22: BlockingArrayQueue (*) => 352 nanoseconds/message (58.67x)
 * 23: PooledBlockingArrayQueue (*) => 430 nanoseconds/message (71.67x)
 * 
 * (*) => Non-batching mode
 * 
 * Test details:
 * 
 * Warmup iterations per queue: 20
 * Outliers ignored (worst and best): 10
 * Total number of iterations: 60 (considered: 30)
 * Total number of messages sent to the other thread: 10,000,000
 * Queue buffer size: 1,048,576
 * Also test non-batching queues: true
 * </pre>
 * <pre>
 * ******************************************************************
 *   LINUX - Two Physical Processors
 * ******************************************************************
 * 
 * 2 x Intel(R) Xeon(R) CPU X5570 @ 2.93GHz
 * Two physical processor with 8 logical processors
 * Java HotSpot(TM) 64-Bit Server VM (build 20.6-b01, mixed mode)
 * Linux solievira-linux 2.6.32-220.7.1.el6.x86_64 x86_64 GNU/Linux
 * 
 * Final results:
 * 
 * 1: BrokenFastQueue => 1 nanoseconds/message
 * 2: <b>AtomicQueue => 9 nanoseconds/message</b>
 * 3: VolatileQueue => 14 nanoseconds/message (1.56x)
 * 4: SynchronizedQueue => 124 nanoseconds/message (13.78x)
 * 5: AtomicQueue (*) => 151 nanoseconds/message (16.78x)
 * 6: ConcurrentLinkedQueue => 168 nanoseconds/message (18.67x)
 * 7: VolatileQueue (*) => 179 nanoseconds/message (19.89x)
 * 8: LockedQueue => 273 nanoseconds/message (30.33x)
 * 9: SynchronizedQueue (*) => 312 nanoseconds/message (34.67x)
 * 10: PooledConcurrentLinkedQueue => 340 nanoseconds/message (37.78x)
 * 11: BlockingQueue => 374 nanoseconds/message (41.56x)
 * 12: BlockingQueue (*) => 541 nanoseconds/message (60.11x)
 * 13: PooledBlockingLinkedQueue (*) => 604 nanoseconds/message (67.11x)
 * 14: PooledBlockingLinkedQueue => 605 nanoseconds/message (67.22x)
 * 15: LockedQueue (*) => 607 nanoseconds/message (67.44x)
 * 16: BlockingArrayQueue => 630 nanoseconds/message (70.0x)
 * 17: BlockingLinkedQueue => 631 nanoseconds/message (70.11x)
 * 18: PooledBlockingQueue => 641 nanoseconds/message (71.22x)
 * 19: PooledBlockingArrayQueue => 644 nanoseconds/message (71.56x)
 * 20: PooledBlockingArrayQueue (*) => 698 nanoseconds/message (77.56x)
 * 21: BlockingLinkedQueue (*) => 712 nanoseconds/message (79.11x)
 * 22: PooledBlockingQueue (*) => 745 nanoseconds/message (82.78x)
 * 23: BlockingArrayQueue (*) => 810 nanoseconds/message (90.0x)
 * 
 * (*) => Non-batching mode
 * 
 * Test details:
 * 
 * Warmup iterations per queue: 20
 * Outliers ignored (worst and best): 10
 * Total number of iterations: 50 (considered: 20)
 * Total number of messages sent to the other thread: 1,000,000
 * Queue buffer size: 1,048,576
 * Also test non-batching queues: true
 * </pre> 
 * <pre>
 * ******************************************************************
 *   MAC OSX
 * ******************************************************************
 * 
 * MacBook Pro Intel Core 2 Duo 2.53 GHz
 * One physical processor with 2 logical processors
 * Java HotSpot(TM) 64-Bit Server VM (build 20.4-b02-402, mixed mode)
 * Darwin soliveira.local 10.8.0 Darwin Kernel Version 10.8.0 root:xnu-1504.15.3~1/RELEASE_I386 i386
 * 
 * Final results:
 * 
 * 1: BrokenFastQueue => 2 nanoseconds/message
 * 2: <b>AtomicQueue => 16 nanoseconds/message</b>
 * 3: AtomicQueue (*) => 26 nanoseconds/message (1.63x)
 * 4: VolatileQueue => 27 nanoseconds/message (1.69x)
 * 5: SynchronizedQueue => 35 nanoseconds/message (2.19x)
 * 6: LockedQueue => 53 nanoseconds/message (3.31x)
 * 7: VolatileQueue (*) => 60 nanoseconds/message (3.75x)
 * 8: ConcurrentLinkedQueue => 88 nanoseconds/message (5.5x)
 * 9: SynchronizedQueue (*) => 118 nanoseconds/message (7.38x)
 * 10: PooledConcurrentLinkedQueue => 150 nanoseconds/message (9.38x)
 * 11: LockedQueue (*) => 152 nanoseconds/message (9.5x)
 * 12: BlockingLinkedQueue => 237 nanoseconds/message (14.81x)
 * 13: BlockingLinkedQueue (*) => 241 nanoseconds/message (15.06x)
 * 14: PooledBlockingLinkedQueue => 250 nanoseconds/message (15.63x)
 * 15: BlockingArrayQueue => 266 nanoseconds/message (16.63x)
 * 16: BlockingQueue => 267 nanoseconds/message (16.69x)
 * 17: PooledBlockingQueue (*) => 270 nanoseconds/message (16.88x)
 * 18: BlockingQueue (*) => 273 nanoseconds/message (17.06x)
 * 19: PooledBlockingLinkedQueue (*) => 275 nanoseconds/message (17.19x)
 * 20: PooledBlockingArrayQueue => 287 nanoseconds/message (17.94x)
 * 21: PooledBlockingQueue => 297 nanoseconds/message (18.56x)
 * 22: BlockingArrayQueue (*) => 361 nanoseconds/message (22.56x)
 * 23: PooledBlockingArrayQueue (*) => 371 nanoseconds/message (23.19x)
 * 
 * (*) => Non-batching mode
 * 
 * Test details:
 * 
 * Warmup iterations per queue: 20
 * Outliers ignored (worst and best): 10
 * Total number of iterations: 50 (considered: 20)
 * Total number of messages sent to the other thread: 1,000,000
 * Queue buffer size: 1,048,576
 * Also test non-batching queues: true
 * 
 * </pre>
 * </p>
 * @author Sergio Oliveira Jr.
 */
public class TestAll {

	private static final Map<Long, String> results = new TreeMap<Long, String>();
	
	private static final NumberFormat NUMBER_FORMATTER = new DecimalFormat("#,###,###");

	// We don't want to overwrite results with same value!
	private static final void addResult(long value, String res) {
		while(results.containsKey(value)) {
			value--; // make it worse to be fair...
		}
		results.put(value, res);
	}
	
	public static void main(String[] args) throws Exception {
		
		if (args.length != 5) {
			System.out.println("format: java -Xms1g -Xmx4g -XX:NewSize=512m -XX:MaxNewSize=1024m -DtestNonBatchingQueues=true -cp target/classes org.mentaqueue.test.producer.TestAll IGNORE OUTLIERS RUNS MESSAGES BUFFER_SIZE");
			System.out.println("IGNORE => How many initial runs will be ignored when calculating the average (warmup runs)?");
			System.out.println("OUTLIERS => How many best and worst to remove?");
			System.out.println("RUNS => How many times to run the test?");
			System.out.println("MESSAGES => How many messages to send to the other thread? (in thousands)");
			System.out.println("BUFFER_SIZE => How many messages can the buffer hold? (in multiples of 1024)");
			System.out.println();
			return;
		}
		
		String[] params = new String[6];
		for(int i = 0; i < args.length; i++) {
			params[i] = args[i];
		}
		params[5] = "true";

		String s = System.getProperty("testNonBatchingQueues");
		boolean testNonBatching = false;
		if (s != null && s.equals("true")) {
			testNonBatching = true;
		}
		
		for(int i = 0; i < 2; i++) {
			
			if (i == 1) {
				if (!testNonBatching) {
					break;
				} else {
					System.out.println();
					System.out.println("Now testing queues on non-batching mode:");
					System.out.println();
				}
			}
			
			String suffix = "";
			
			if (i == 1 && testNonBatching) {
				suffix = " (*)";
				params[5] = "false";
			}
		
			if (i == 0) {
				TestBrokenFastQueue.main(params);
				addResult(TestBrokenFastQueue.getResult(), BrokenFastQueue.class.getSimpleName() + suffix);
			}
			
			TestAtomicQueue.main(params);
			addResult(TestAtomicQueue.getResult(), AtomicQueue.class.getSimpleName() + suffix);
			
			TestVolatileQueue.main(params);
			addResult(TestVolatileQueue.getResult(), VolatileQueue.class.getSimpleName() + suffix);
			
			TestLockedQueue.main(params);
			addResult(TestLockedQueue.getResult(), LockedQueue.class.getSimpleName() + suffix);
			
			TestSynchronizedQueue.main(params);
			addResult(TestSynchronizedQueue.getResult(), SynchronizedQueue.class.getSimpleName() + suffix);
			
			if (i == 0) {
				TestPooledConcurrentLinkedQueue.main(params);
				addResult(TestPooledConcurrentLinkedQueue.getResult(), PooledConcurrentLinkedQueue.class.getSimpleName() + suffix);
				
				TestConcurrentLinkedQueue.main(params);
				addResult(TestConcurrentLinkedQueue.getResult(), ConcurrentLinkedQueue.class.getSimpleName() + suffix);
			}
	
			TestPooledBlockingArrayQueue.main(params);
			addResult(TestPooledBlockingArrayQueue.getResult(), PooledBlockingArrayQueue.class.getSimpleName() + suffix);
			
			TestPooledBlockingLinkedQueue.main(params);
			addResult(TestPooledBlockingLinkedQueue.getResult(), PooledBlockingLinkedQueue.class.getSimpleName() + suffix);
			
			TestBlockingLinkedQueue.main(params);
			addResult(TestBlockingLinkedQueue.getResult(), BlockingLinkedQueue.class.getSimpleName() + suffix);
			
			TestBlockingArrayQueue.main(params);
			addResult(TestBlockingArrayQueue.getResult(), BlockingArrayQueue.class.getSimpleName() + suffix);
			
			TestBlockingQueue.main(params);
			addResult(TestBlockingQueue.getResult(), BlockingQueue.class.getSimpleName() + suffix);
			
			TestPooledBlockingQueue.main(params);
			addResult(TestPooledBlockingQueue.getResult(), PooledBlockingQueue.class.getSimpleName() + suffix);
		}
		
		System.out.println();
		System.out.println("Final results:\n");
		
		Iterator<Long> iter = results.keySet().iterator();
		
		int count = 1;
		
		long champion = 0;
		
		while(iter.hasNext()) {
			long nanosPerMsg = iter.next();
			String queue = results.get(nanosPerMsg);
			String res = count + ": " + queue + " => " + NUMBER_FORMATTER.format(nanosPerMsg) + " nanoseconds/message";
			
			if (count == 2) {
				// first one must be BrokenFastQueue
				champion = nanosPerMsg;
			} else if (count > 2) {
				res += " (" + roundTwoDecimals(nanosPerMsg, champion) + "x)";
			}
			
			System.out.println(res);
			
			count++;
		}
		
		System.out.println();
		if (testNonBatching) {
			System.out.println("(*) => Non-batching mode");
			System.out.println();
		}
		
		System.out.println("Test details:");
		System.out.println();
		
		System.out.println("Warmup iterations per queue: " + args[0]);
		System.out.println("Outliers ignored (worst and best): " + Integer.parseInt(args[1]) * 2);
		
		int considered = Integer.parseInt(args[2]) - (Integer.parseInt(args[0]) + Integer.parseInt(args[1]) * 2);
		
		System.out.println("Total number of iterations: " + args[2] + " (considered: " + considered + ")");
		System.out.println("Total number of messages sent to the other thread: " + NUMBER_FORMATTER.format(Integer.parseInt(args[3]) * 1000));
		System.out.println("Queue buffer size: " + NUMBER_FORMATTER.format(Integer.parseInt(args[4]) * 1024));
		System.out.println("Also test non-batching queues: " + testNonBatching);
		System.out.println();
	}
	
	private static String roundTwoDecimals(long x, long y) {
		float f = (float) x / (float) y;
		f *= 100f;
		f = Math.round(f);
		f /= 100f;
		return String.valueOf(f);
	}
}
