/*
 * MentaQueue => http://mentaqueue.soliveirajr.com
 * Copyright (C) 2012  Sergio Oliveira Jr. (sergio.oliveira.jr@gmail.com)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package org.mentaqueue.test.messages;

import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentLinkedQueue;

import org.mentaqueue.AtomicQueue;
import org.mentaqueue.BlockingArrayQueue;
import org.mentaqueue.BlockingLinkedQueue;
import org.mentaqueue.BrokenFastQueue;
import org.mentaqueue.LockedQueue;
import org.mentaqueue.SynchronizedQueue;
import org.mentaqueue.VolatileQueue;
import org.mentaqueue.pooled.PooledBlockingArrayQueue;
import org.mentaqueue.pooled.PooledBlockingLinkedQueue;
import org.mentaqueue.pooled.PooledBlockingQueue;
import org.mentaqueue.pooled.PooledConcurrentLinkedQueue;
/**
 * <p>
 * To run: 
 * <pre>
 * java -Xms1g -Xmx4g -XX:NewSize=512m -XX:MaxNewSize=1024m -DtestNonBatchingQueues=true -cp target/classes org.mentaqueue.test.messages.TestAll 20 5 50 1000 1024
 * 
 * format: java -Xms1g -Xmx4g -XX:NewSize=512m -XX:MaxNewSize=1024m -DtestUncheckedQueues=true -DtestNonBatchingQueues=true -cp target/classes org.mentaqueue.test.messages.TestAll IGNORE OUTLIERS RUNS MESSAGES BUFFER_SIZE
 * IGNORE => How many initial runs will be ignored when calculating the average (warmup runs)?
 * OUTLIERS => How many best and worst to remove?
 * RUNS => How many times to run the test?
 * MESSAGES => How many messages to send to the other thread? (in thousands)
 * BUFFER_SIZE => How many messages can the buffer hold? (in multiples of 1024)
 * </pre>
 * </p>
 * <p>
 * Latest results:
 * <pre>
 * ******************************************************************
 *   LINUX - One Physical Processor
 * ******************************************************************
 * 
 * Intel(R) Xeon(R) CPU E5345 @ 2.33GHz
 * One physical processor with 6 logical processors
 * Java HotSpot(TM) 64-Bit Server VM (build 20.2-b06, mixed mode)
 * Linux soliveira-linux 2.6.32-30-generic x86_64 GNU/Linux
 * 
 * Final results:
 * 
 * 1: BrokenFastQueue => 1,416,666,666 messages/sec
 * 2: <b>AtomicQueue => 138,133,212 messages/sec</b>
 * 3: VolatileQueue => 90,336,277 messages/sec (1.53x)
 * 4: LockedQueue => 25,278,134 messages/sec (5.46x)
 * 5: AtomicQueue (*) => 21,829,598 messages/sec (6.33x)
 * 6: SynchronizedQueue => 14,059,058 messages/sec (9.83x)
 * 7: ConcurrentLinkedQueue => 13,392,916 messages/sec (10.31x)
 * 8: VolatileQueue (*) => 13,367,912 messages/sec (10.33x)
 * 9: PooledConcurrentLinkedQueue => 6,802,307 messages/sec (20.31x)
 * 10: SynchronizedQueue (*) => 5,668,692 messages/sec (24.37x)
 * 11: PooledBlockingQueue => 5,662,282 messages/sec (24.4x)
 * 12: BlockingQueue => 5,289,564 messages/sec (26.11x)
 * 13: BlockingLinkedQueue => 4,151,632 messages/sec (33.27x)
 * 14: PooledBlockingLinkedQueue => 3,971,971 messages/sec (34.78x)
 * 15: BlockingLinkedQueue (*) => 3,909,367 messages/sec (35.33x)
 * 16: BlockingQueue (*) => 3,730,679 messages/sec (37.03x)
 * 17: BlockingArrayQueue => 3,568,064 messages/sec (38.71x)
 * 18: PooledBlockingLinkedQueue (*) => 3,546,631 messages/sec (38.95x)
 * 19: LockedQueue (*) => 3,350,480 messages/sec (41.23x)
 * 20: PooledBlockingArrayQueue => 3,044,092 messages/sec (45.38x)
 * 21: PooledBlockingQueue (*) => 3,038,979 messages/sec (45.45x)
 * 22: BlockingArrayQueue (*) => 2,972,741 messages/sec (46.47x)
 * 23: PooledBlockingArrayQueue (*) => 2,347,767 messages/sec (58.84x)
 * 
 * (*) => Non-batching mode
 * 
 * Test details:
 * 
 * Warmup iterations per queue: 20
 * Outliers ignored (worst and best): 10
 * Total number of iterations: 60 (considered: 30)
 * Total number of messages sent to the other thread: 10,000,000
 * Queue buffer size: 1,048,576
 * Also test non-batching queues: true
 * 
 * </pre>
 * <pre>
 * ******************************************************************
 *   LINUX - Two Physical Processors
 * ******************************************************************
 * 
 * 2 x Intel(R) Xeon(R) CPU X5570 @ 2.93GHz
 * Two physical processor with 8 logical processors
 * Java HotSpot(TM) 64-Bit Server VM (build 20.6-b01, mixed mode)
 * Linux solievira-linux 2.6.32-220.7.1.el6.x86_64 x86_64 GNU/Linux
 * 
 * Final results:
 * 
 * 1: BrokenFastQueue => 500,000,000 messages/sec
 * 2: <b>AtomicQueue => 90,909,090 messages/sec</b>
 * 3: VolatileQueue => 62,500,000 messages/sec (1.45x)
 * 4: SynchronizedQueue => 9,260,470 messages/sec (9.82x)
 * 5: AtomicQueue (*) => 6,925,585 messages/sec (13.13x)
 * 6: LockedQueue => 6,027,541 messages/sec (15.08x)
 * 7: ConcurrentLinkedQueue => 5,770,940 messages/sec (15.75x)
 * 8: VolatileQueue (*) => 5,391,251 messages/sec (16.86x)
 * 9: SynchronizedQueue (*) => 3,405,292 messages/sec (26.7x)
 * 10: PooledConcurrentLinkedQueue => 2,838,052 messages/sec (32.03x)
 * 11: BlockingQueue => 2,625,946 messages/sec (34.62x)
 * 12: BlockingQueue (*) => 2,394,405 messages/sec (37.97x)
 * 13: PooledBlockingLinkedQueue => 1,606,232 messages/sec (56.6x)
 * 14: BlockingLinkedQueue => 1,590,376 messages/sec (57.16x)
 * 15: PooledBlockingArrayQueue => 1,500,989 messages/sec (60.57x)
 * 16: LockedQueue (*) => 1,479,300 messages/sec (61.45x)
 * 17: PooledBlockingQueue => 1,449,377 messages/sec (62.72x)
 * 18: BlockingArrayQueue => 1,408,334 messages/sec (64.55x)
 * 19: BlockingLinkedQueue (*) => 1,401,242 messages/sec (64.88x)
 * 20: PooledBlockingArrayQueue (*) => 1,321,562 messages/sec (68.79x)
 * 21: PooledBlockingLinkedQueue (*) => 1,314,472 messages/sec (69.16x)
 * 22: BlockingArrayQueue (*) => 1,247,140 messages/sec (72.89x)
 * 23: PooledBlockingQueue (*) => 1,167,021 messages/sec (77.9x)
 * 
 * (*) => Non-batching mode
 * 
 * Test details:
 * 
 * Warmup iterations per queue: 20
 * Outliers ignored (worst and best): 10
 * Total number of iterations: 50 (considered: 20)
 * Total number of messages sent to the other thread: 1,000,000
 * Queue buffer size: 1,048,576
 * Also test non-batching queues: true
 * 
 * </pre> 
 * <pre>
 * ******************************************************************
 *   MAC OSX
 * ******************************************************************
 * 
 * MacBook Pro Intel Core 2 Duo 2.53 GHz
 * One physical processor with 2 logical processors
 * Java HotSpot(TM) 64-Bit Server VM (build 20.4-b02-402, mixed mode)
 * Darwin soliveira.local 10.8.0 Darwin Kernel Version 10.8.0 root:xnu-1504.15.3~1/RELEASE_I386 i386
 * 
 * Final results:
 * 
 * 1: BrokenFastQueue => 500,000,000 messages/sec
 * 2: <b>AtomicQueue => 55,116,958 messages/sec</b>
 * 3: VolatileQueue => 33,513,162 messages/sec (1.64x)
 * 4: AtomicQueue (*) => 28,655,461 messages/sec (1.92x)
 * 5: SynchronizedQueue => 26,991,465 messages/sec (2.04x)
 * 6: VolatileQueue (*) => 19,823,529 messages/sec (2.78x)
 * 7: LockedQueue => 18,451,178 messages/sec (2.99x)
 * 8: ConcurrentLinkedQueue => 10,900,873 messages/sec (5.06x)
 * 9: PooledConcurrentLinkedQueue => 6,377,823 messages/sec (8.64x)
 * 10: SynchronizedQueue (*) => 5,756,309 messages/sec (9.58x)
 * 11: LockedQueue (*) => 4,662,285 messages/sec (11.82x)
 * 12: BlockingLinkedQueue => 4,111,964 messages/sec (13.4x)
 * 13: BlockingLinkedQueue (*) => 3,978,599 messages/sec (13.85x)
 * 14: PooledBlockingLinkedQueue => 3,833,007 messages/sec (14.38x)
 * 15: BlockingQueue (*) => 3,659,262 messages/sec (15.06x)
 * 16: PooledBlockingLinkedQueue (*) => 3,645,013 messages/sec (15.12x)
 * 17: BlockingArrayQueue => 3,610,799 messages/sec (15.26x)
 * 18: BlockingQueue => 3,427,360 messages/sec (16.08x)
 * 19: PooledBlockingArrayQueue => 3,286,787 messages/sec (16.77x)
 * 20: PooledBlockingQueue (*) => 3,161,825 messages/sec (17.43x)
 * 21: PooledBlockingQueue => 3,118,518 messages/sec (17.67x)
 * 22: PooledBlockingArrayQueue (*) => 2,695,151 messages/sec (20.45x)
 * 23: BlockingArrayQueue (*) => 2,682,131 messages/sec (20.55x)
 * 
 * (*) => Non-batching mode
 * 
 * Test details:
 * 
 * Warmup iterations per queue: 20
 * Outliers ignored (worst and best): 10
 * Total number of iterations: 50 (considered: 20)
 * Total number of messages sent to the other thread: 1,000,000
 * Queue buffer size: 1,048,576
 * Also test non-batching queues: true
 * 
 * </pre>
 * </p>
 * @author Sergio Oliveira Jr.
 */
public class TestAll {

	private static final Map<Long, String> results = new TreeMap<Long, String>(Collections.reverseOrder());
	
	private static final NumberFormat NUMBER_FORMATTER = new DecimalFormat("#,###,###");

	// We don't want to overwrite results with same value!
	private static final void addResult(long value, String res) {
		while(results.containsKey(value)) {
			value--; // make it worse to be fair...
		}
		results.put(value, res);
	}
	
	public static void main(String[] args) throws Exception {
		
		if (args.length != 5) {
			System.out.println("format: java -Xms1g -Xmx4g -XX:NewSize=512m -XX:MaxNewSize=1024m -DtestNonBatchingQueues=true -cp target/classes org.mentaqueue.test.messages.TestAll IGNORE OUTLIERS RUNS MESSAGES BUFFER_SIZE");
			System.out.println("IGNORE => How many initial runs will be ignored when calculating the average (warmup runs)?");
			System.out.println("OUTLIERS => How many best and worst to remove?");
			System.out.println("RUNS => How many times to run the test?");
			System.out.println("MESSAGES => How many messages to send to the other thread? (in thousands)");
			System.out.println("BUFFER_SIZE => How many messages can the buffer hold? (in multiples of 1024)");
			System.out.println();
			return;
		}
		
		String[] params = new String[6];
		for(int i = 0; i < args.length; i++) {
			params[i] = args[i];
		}
		params[5] = "true";

		String s = System.getProperty("testNonBatchingQueues");
		boolean testNonBatching = false;
		if (s != null && s.equals("true")) {
			testNonBatching = true;
		}
		
		for(int i = 0; i < 2; i++) {
			
			if (i == 1) {
				if (!testNonBatching) {
					break;
				} else {
					System.out.println();
					System.out.println("Now testing queues on non-batching mode:");
					System.out.println();
				}
			}
			
			String suffix = "";
			
			if (i == 1 && testNonBatching) {
				suffix = " (*)";
				params[5] = "false";
			}
		
			if (i == 0) {
				TestBrokenFastQueue.main(params);
				addResult(TestBrokenFastQueue.getResult(), BrokenFastQueue.class.getSimpleName() + suffix);
			}
			
			TestAtomicQueue.main(params);
			addResult(TestAtomicQueue.getResult(), AtomicQueue.class.getSimpleName() + suffix);
			
			TestVolatileQueue.main(params);
			addResult(TestVolatileQueue.getResult(), VolatileQueue.class.getSimpleName() + suffix);
			
			TestLockedQueue.main(params);
			addResult(TestLockedQueue.getResult(), LockedQueue.class.getSimpleName() + suffix);
			
			TestSynchronizedQueue.main(params);
			addResult(TestSynchronizedQueue.getResult(), SynchronizedQueue.class.getSimpleName() + suffix);
			
			if (i == 0) {
				TestPooledConcurrentLinkedQueue.main(params);
				addResult(TestPooledConcurrentLinkedQueue.getResult(), PooledConcurrentLinkedQueue.class.getSimpleName() + suffix);
				
				TestConcurrentLinkedQueue.main(params);
				addResult(TestConcurrentLinkedQueue.getResult(), ConcurrentLinkedQueue.class.getSimpleName() + suffix);
			}
	
			TestPooledBlockingArrayQueue.main(params);
			addResult(TestPooledBlockingArrayQueue.getResult(), PooledBlockingArrayQueue.class.getSimpleName() + suffix);
			
			TestPooledBlockingLinkedQueue.main(params);
			addResult(TestPooledBlockingLinkedQueue.getResult(), PooledBlockingLinkedQueue.class.getSimpleName() + suffix);
			
			TestBlockingLinkedQueue.main(params);
			addResult(TestBlockingLinkedQueue.getResult(), BlockingLinkedQueue.class.getSimpleName() + suffix);
			
			TestBlockingArrayQueue.main(params);
			addResult(TestBlockingArrayQueue.getResult(), BlockingArrayQueue.class.getSimpleName() + suffix);
			
			TestBlockingQueue.main(params);
			addResult(TestBlockingQueue.getResult(), BlockingQueue.class.getSimpleName() + suffix);
			
			TestPooledBlockingQueue.main(params);
			addResult(TestPooledBlockingQueue.getResult(), PooledBlockingQueue.class.getSimpleName() + suffix);
		}
		
		System.out.println();
		System.out.println("Final results:\n");
		
		Iterator<Long> iter = results.keySet().iterator();
		
		int count = 1;
		
		long champion = 0;
		
		while(iter.hasNext()) {
			long messagesPerSec = iter.next();
			String queue = results.get(messagesPerSec);
			String res = count + ": " + queue + " => " + NUMBER_FORMATTER.format(messagesPerSec) + " messages/sec";
			
			if (count == 2) {
				// first one must be BrokenFastQueue
				champion = messagesPerSec;
			} else if (count > 2) {
				res += " (" + roundTwoDecimals(champion, messagesPerSec) + "x)";
			}
			
			System.out.println(res);
			
			count++;
		}
		
		System.out.println();
		if (testNonBatching) {
			System.out.println("(*) => Non-batching mode");
			System.out.println();
		}
		
		System.out.println("Test details:");
		System.out.println();
		
		System.out.println("Warmup iterations per queue: " + args[0]);
		System.out.println("Outliers ignored (worst and best): " + Integer.parseInt(args[1]) * 2);
		
		int considered = Integer.parseInt(args[2]) - (Integer.parseInt(args[0]) + Integer.parseInt(args[1]) * 2);
		
		System.out.println("Total number of iterations: " + args[2] + " (considered: " + considered + ")");
		System.out.println("Total number of messages sent to the other thread: " + NUMBER_FORMATTER.format(Integer.parseInt(args[3]) * 1000));
		System.out.println("Queue buffer size: " + NUMBER_FORMATTER.format(Integer.parseInt(args[4]) * 1024));
		System.out.println("Also test non-batching queues: " + testNonBatching);
		System.out.println();
	}
	
	private static String roundTwoDecimals(long x, long y) {
		float f = (float) x / (float) y;
		f *= 100f;
		f = Math.round(f);
		f /= 100f;
		return String.valueOf(f);
	}
}
