/*
 * MentaQueue => http://mentaqueue.soliveirajr.com
 * Copyright (C) 2012  Sergio Oliveira Jr. (sergio.oliveira.jr@gmail.com)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package org.mentaqueue.pooled;

import static org.mentaqueue.util.BuilderUtils.createBuilder;

import org.mentaqueue.BatchingQueue;
import org.mentaqueue.util.Builder;

public class PooledConcurrentLinkedQueue<E> implements BatchingQueue<E> {

	private final static int DEFAULT_CAPACITY = 1026 * 16;

	private final java.util.concurrent.ConcurrentLinkedQueue<E> pool;
	private final java.util.concurrent.ConcurrentLinkedQueue<E> queue;
	private final Builder<E> builder;
	private E returnedElem = null;
	
    @SuppressWarnings("unchecked")
    private final E[] temp = (E[]) new Object[DEFAULT_CAPACITY];
    private int tempPointer = 0;

	public PooledConcurrentLinkedQueue(int capacity, Builder<E> builder) {
		pool = new java.util.concurrent.ConcurrentLinkedQueue<E>();
		queue = new java.util.concurrent.ConcurrentLinkedQueue<E>();
		this.builder = builder;
		for (int i = 0; i < capacity; i++) {
			pool.offer(builder.newInstance());
		}
	}

	public PooledConcurrentLinkedQueue(Builder<E> builder) {
		this(DEFAULT_CAPACITY, builder);
	}
	
	public PooledConcurrentLinkedQueue(Class<E> klass) {
		this(createBuilder(klass));
	}
	
	public PooledConcurrentLinkedQueue(int capacity, Class<E> klass) {
		this(capacity, createBuilder(klass));
	}


	@Override
	public E nextToDispatch() {
		E e = pool.poll();
		if (e == null) {
			e = builder.newInstance();
		}
		temp[tempPointer++] = e;
		return e;
	}
	
	@Override
	public final void flush() {
		flush(false);
	}

	@Override
	public void flush(boolean lazySet) {
		for(int i = 0; i < tempPointer; i++) {
			queue.offer(temp[i]);
			temp[i] = null;
		}
		tempPointer = 0;
	}

	@Override
	public long availableToPoll() {
		E e = queue.peek();
		return e != null ? 1 : 0;
	}

	@Override
	public E poll() {
		returnedElem = queue.poll();
		return returnedElem;
	}
	
	@Override
	public E peek() {
		return queue.peek();
	}

	@Override
	public void donePolling(boolean lazySet) {
		pool.offer(returnedElem);
		returnedElem = null;
	}
	
	@Override
	public final void donePolling() {
		donePolling(false);
	}
	
	@Override
	public final void rollback() {
		throw new UnsupportedOperationException();
	}
	
	@Override
	public final void rollback(long count) {
		throw new UnsupportedOperationException();
	}
}
