/*
 * MentaQueue => http://mentaqueue.soliveirajr.com
 * Copyright (C) 2012  Sergio Oliveira Jr. (sergio.oliveira.jr@gmail.com)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package org.mentaqueue.pooled;

import static org.mentaqueue.util.BuilderUtils.createBuilder;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.mentaqueue.BatchingQueue;
import org.mentaqueue.util.Builder;

public class PooledBlockingArrayQueue<E> implements BatchingQueue<E> {

	private final static int DEFAULT_CAPACITY = 1026 * 16;

	private final java.util.concurrent.ConcurrentLinkedQueue<E> pool;
	private final List<E> polledElements;
	private final java.util.concurrent.ArrayBlockingQueue<E> queue;
	private final Builder<E> builder;
	
    @SuppressWarnings("unchecked")
    private final E[] temp = (E[]) new Object[DEFAULT_CAPACITY];
    private int tempPointer = 0;

	public PooledBlockingArrayQueue(int capacity, Builder<E> builder) {
		pool = new java.util.concurrent.ConcurrentLinkedQueue<E>();
		polledElements = new LinkedList<E>();
		queue = new java.util.concurrent.ArrayBlockingQueue<E>(capacity);
		this.builder = builder;
		for (int i = 0; i < capacity; i++) {
			pool.offer(builder.newInstance());
		}
	}

	public PooledBlockingArrayQueue(Builder<E> builder) {
		this(DEFAULT_CAPACITY, builder);
	}
	
	public PooledBlockingArrayQueue(Class<E> klass) {
		this(createBuilder(klass));
	}
	
	public PooledBlockingArrayQueue(int capacity, Class<E> klass) {
		this(capacity, createBuilder(klass));
	}


	@Override
	public E nextToDispatch() {
		E e = pool.poll();
		if (e == null) {
			e = builder.newInstance();
		}
		temp[tempPointer++] = e;
		return e;
	}

	@Override
	public void flush(boolean lazySet) {
		try {
			for(int i = 0; i < tempPointer; i++) {
				queue.put(temp[i]);
				temp[i] = null;
			}
			tempPointer = 0;
		} catch(Exception exc) {
			throw new RuntimeException(exc);
		}
	}
	
	@Override
	public final void flush() {
		flush(false);
	}

	@Override
	public long availableToPoll() {
		return queue.size();
	}

	@Override
	public E poll() {
		E returnedElem = queue.poll();
		polledElements.add(returnedElem);
		return returnedElem;
	}
	
	@Override
	public E peek() {
		return queue.peek();
	}

	@Override
	public void donePolling(boolean lazySet) {
		Iterator<E> iter = polledElements.iterator();
		while(iter.hasNext()) {
			pool.offer(iter.next());
			iter.remove();
		}
	}
	
	@Override
	public final void donePolling() {
		donePolling(false);
	}
	
	@Override
	public final void rollback() {
		throw new UnsupportedOperationException();
	}
	
	@Override
	public final void rollback(long count) {
		throw new UnsupportedOperationException();
	}
}
