/*
 * MentaQueue => http://mentaqueue.soliveirajr.com
 * Copyright (C) 2012  Sergio Oliveira Jr. (sergio.oliveira.jr@gmail.com)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package org.mentaqueue;

import static org.mentaqueue.util.BuilderUtils.createBuilder;

import org.mentaqueue.util.Builder;
import org.mentaqueue.util.NumberUtils;

public class VolatileQueue<E> implements BatchingQueue<E> {

	private final static int DEFAULT_CAPACITY = 1024 * 16;

	private final int capacity;
	private final E[] data;
	private long nextOfferValue = -1;
	private long nextPollValue = -1;
	private long pollCounter = 0;
	private volatile long sequence = -1;
	private volatile long pollingSequence = -1;
	private long maxSeq;

	@SuppressWarnings("unchecked")
	public VolatileQueue(int capacity, Builder<E> builder) {
		NumberUtils.ensurePowerOfTwo(capacity);
		this.capacity = capacity;
		this.data = (E[]) new Object[capacity];
		for (int i = 0; i < capacity; i++) {
			this.data[i] = builder.newInstance();
		}
		this.maxSeq = findMaxSeqBeforeWrapping();
	}

	public VolatileQueue(Builder<E> builder) {
		this(DEFAULT_CAPACITY, builder);
	}
	
	public VolatileQueue(Class<E> klass) {
		this(createBuilder(klass));
	}
	
	public VolatileQueue(int capacity, Class<E> klass) {
		this(capacity, createBuilder(klass));
	}

	
	private final long findMaxSeqBeforeWrapping() {
		return capacity + pollingSequence;
	}

	@Override
	public E nextToDispatch() {
		long index = ++nextOfferValue;
		if (index > maxSeq) {
			// this would wrap the buffer... calculate the new one...
			this.maxSeq = findMaxSeqBeforeWrapping();
			if (index > maxSeq) {
				nextOfferValue--;
				return null;				
			}
		}
		return data[(int) (index & capacity - 1)];
	}

	@Override
	public void flush(boolean lazySet) {
		sequence = nextOfferValue;
	}
	
	@Override
	public final void flush() {
		flush(false);
	}

	@Override
	public long availableToPoll() {
		return sequence - nextPollValue;
	}

	@Override
	public E poll() {
		pollCounter++;
		return data[(int) (++nextPollValue & capacity - 1)];
	}
	
	@Override
	public E peek() {
		return data[(int) (nextPollValue & capacity - 1)];
	}

	@Override
	public void donePolling(boolean lazySet) {
		pollingSequence = nextPollValue;
		pollCounter = 0;
	}
	
	@Override
	public final void donePolling() {
		donePolling(false);
	}
	
	@Override
	public final void rollback() {
		rollback(pollCounter);
	}
	
	@Override
	public final void rollback(long count) {
		if (count < 0 || count > pollCounter) {
			throw new RuntimeException("Invalid rollback request! polled=" + pollCounter + " requested=" + count);
		}
		nextPollValue -= count;
		pollCounter -= count;
	}
}