/*
 * MentaQueue => http://mentaqueue.soliveirajr.com
 * Copyright (C) 2012  Sergio Oliveira Jr. (sergio.oliveira.jr@gmail.com)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package org.mentaqueue;

import java.util.Collection;
import java.util.Iterator;
import java.util.Queue;

/**
 * A non-batching queue wrapper that takes a BatchingQueue and makes it non-batching by implementing only the poll() and offer() methods.
 *
 * Since it implements java.util.Queue you can use this wrapper as a replacement of the standard Java queue implementations.
 * 
 * However you should keep in mind that batching is one of the things that make MentaQueue fast, so a non-batching queue will be many times
 * slower. It is highly recommended that you refactor your code to use the BatchingQueue interface because the performance gains are much larger.
 * 
 * @author Sergio Oliveira Jr.
 *
 * @param <E>
 */
public class NonBatchingQueue<E> implements Queue<E> {
	
	private static final boolean DEFAULT_LAZY_SET = true;
	
	private final boolean isLazySet;
	
	public static class Entry<E> {
		E e;
	}
	
	private final BatchingQueue<Entry<E>> queue;
	
	public NonBatchingQueue(BatchingQueue<Entry<E>> queue) {
		this(queue, DEFAULT_LAZY_SET);
	}
	
	public NonBatchingQueue(BatchingQueue<Entry<E>> queue, boolean isLazySet) {
		this.queue = queue;
		this.isLazySet = isLazySet;
	}

	@Override
    public int size() {
	    throw new UnsupportedOperationException();
    }

	@Override
    public boolean isEmpty() {
		throw new UnsupportedOperationException();
    }

	@Override
    public boolean contains(Object o) {
		throw new UnsupportedOperationException();
    }

	@Override
    public Iterator<E> iterator() {
		throw new UnsupportedOperationException();
    }

	@Override
    public Object[] toArray() {
		throw new UnsupportedOperationException();
    }

	@Override
    public <T> T[] toArray(T[] a) {
		throw new UnsupportedOperationException();
    }

	@Override
    public boolean remove(Object o) {
		throw new UnsupportedOperationException();
    }

	@Override
    public boolean containsAll(Collection<?> c) {
		throw new UnsupportedOperationException();
    }

	@Override
    public boolean addAll(Collection<? extends E> c) {
		throw new UnsupportedOperationException();
    }

	@Override
    public boolean removeAll(Collection<?> c) {
		throw new UnsupportedOperationException();
    }

	@Override
    public boolean retainAll(Collection<?> c) {
		throw new UnsupportedOperationException();
    }

	@Override
    public void clear() {
		throw new UnsupportedOperationException();
    }

	@Override
    public boolean add(E e) {
		throw new UnsupportedOperationException();
    }

	@Override
    public boolean offer(E e) {
		Entry<E> entry = queue.nextToDispatch();
		if (entry != null) {
			entry.e = e;
			queue.flush(isLazySet);
			return true;
		}
		return false;
    }

	@Override
    public E remove() {
		throw new UnsupportedOperationException();
	}

	@Override
    public E poll() {
		if (queue.availableToPoll() > 0) {
			Entry<E> entry = queue.poll();
			E e = entry.e;
			queue.donePolling(isLazySet);
			return e;
		}
		return null;
    }

	@Override
    public E element() {
		throw new UnsupportedOperationException();
	}

	@Override
    public E peek() {
		throw new UnsupportedOperationException();
    }
}
