/*
 * MentaQueue => http://mentaqueue.soliveirajr.com
 * Copyright (C) 2012  Sergio Oliveira Jr. (sergio.oliveira.jr@gmail.com)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package org.mentaqueue;

import static org.mentaqueue.util.BuilderUtils.createBuilder;

import java.util.LinkedList;

import org.mentaqueue.util.Builder;

public class BlockingQueue<E> implements BatchingQueue<E> {

	private final static int DEFAULT_CAPACITY = 1026 * 16;

	private final LinkedList<E> queue;
	private final Object lock = new Object();
	private final int capacity;
	private final Builder<E> builder; 
	
    @SuppressWarnings("unchecked")
    private final E[] temp = (E[]) new Object[DEFAULT_CAPACITY];
    private int tempPointer = 0;

	public BlockingQueue(int capacity, Builder<E> builder) {
		queue = new LinkedList<E>();
		this.capacity = capacity;
		this.builder = builder;
	}

	public BlockingQueue(Builder<E> builder) {
		this(DEFAULT_CAPACITY, builder);
	}
	
	public BlockingQueue(Class<E> klass) {
		this(createBuilder(klass));
	}
	
	public BlockingQueue(int capacity, Class<E> klass) {
		this(capacity, createBuilder(klass));
	}

	@Override
	public E nextToDispatch() {
		E e = builder.newInstance();
		temp[tempPointer++] = e;
		return e;
	}

	@Override
	public void flush(boolean lazySet) {
		synchronized(lock) {
			boolean wasEmpty = queue.isEmpty();
			for(int i = 0; i < tempPointer; i++) {
    			int size = queue.size();
    			while (size == capacity) {
    				try {
    					lock.wait();
    				} catch(Exception exc) {
    					return;
    				}
    			}
    			queue.add(temp[i]);
    			temp[i] = null;
			}
			if (wasEmpty && tempPointer > 0) { // was empty, now it is not...
				lock.notify();
			}
			tempPointer = 0;
		}
	}
	
	@Override
	public final void flush() {
		flush(false);
	}

	@Override
	public long availableToPoll() {
		synchronized(lock) {
			while (queue.size() == 0) {
				try {
					lock.wait();
				} catch(Exception e) {
					return 0;
				}
			}
			return queue.size();
		}
	}

	@Override
	public E poll() {
		synchronized(lock) {
			E elem = queue.removeFirst();
			if (queue.size() == capacity - 1) {
				lock.notify();
			}
			return elem;
		}
	}
	
	@Override
	public E peek() {
		throw new UnsupportedOperationException();
	}

	@Override
	public void donePolling(boolean lazySet) {
	}
	
	@Override
	public final void donePolling() {
		donePolling(false);
	}
	
	@Override
	public final void rollback() {
		throw new UnsupportedOperationException();
	}
	
	@Override
	public final void rollback(long count) {
		throw new UnsupportedOperationException();
	}
}
