/*
 * MentaQueue => http://mentaqueue.soliveirajr.com
 * Copyright (C) 2012  Sergio Oliveira Jr. (sergio.oliveira.jr@gmail.com)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package org.mentaqueue;

/**
 * A queue API that allows batching and pooling of objects.
 * 
 *  So to offer to the queue, you first get a mutable object from the queue by calling <i>nextToDispatch()</i>, alter this object and call <i>flush(boolean lazySet)</i>.
 *  That allows the queue objects to be pooled, avoiding any garbage collection.
 *  
 *  And to poll you first call <i>availableToPoll()</i> to know how many objects you can safely poll, call <i>poll()</i> in a loop and when done call <i>donePolling(boolean lazySet)</i>.
 *  That allows polling to be batched, so you pay a synchronization price only when you call <i>available()</i> and NOT when you call <i>poll()</i>.
 * 
 * @author Sergio Oliveira Jr.
 *
 * @param <E>
 */
public interface BatchingQueue<E> {

	/**
	 * Return the next pooled mutable object that can be used by the producer to dispatch data to the queue.
	 * 
	 * @return the next mutable object that can be used by the producer.
	 */
	public E nextToDispatch();

	/**
	 * Dispatch all previously obtained objects through the <i>nextToDispatch()</i> method to the queue.
	 * 
	 * @param lazySet flush (i.e. notify the consumer) in a lazy way or flush immediately
	 */
	public void flush(boolean lazySet);
	
	/**
	 * Dispatch *immediately* all previously obtained objects through the <i>nextToDispatch()</i> method to the queue.
	 */
	public void flush();
	
	/**
	 * Return the number of objects that can be safely polled from this queue. It can return zero.
	 * 
	 * @return number of objects that can be polled.
	 */
	public long availableToPoll();

	/**
	 * Poll a object from the queue. You can only call this method after calling <i>available()</i> so you
	 * know what is the maximum times you can call it.
	 * 
	 * NOTE: You should NOT keep your own reference for this mutable object. Read what you need to get from it and release its reference.
	 * 
	 * @return an object from the queue.
	 */
	public E poll();

	/**
	 * Called to indicate that all polling have been concluded.
	 * 
	 * @param lazySet notify the producer in a lazy way or notify the producer immediately
	 */
	public void donePolling(boolean lazySet);
	
	/**
	 * Called to indicate that all polling have been concluded.
	 */
	public void donePolling();
	
	/**
	 * Pretend you never polled the last object you polled since the last time you called donePolling().
	 * You can call this as many times as you want before you call donePolling() and rollback any poll() you have done.
	 * This is unaffected by availableToPoll(). Only donePolling() reset the counter of the last polled objects.
	 * Because rollback() reset the counter of the last polled objects, you can even call it twice in a row and the second
	 * rollback() will have no effect since you have polled anything.
	 */
	public void rollback();
	
	/**
	 * Same as rollback but allows you to specify how many previous polls you want to rollback
	 * 
	 * @param items how many polls to rollback
	 */
	public void rollback(long items);
	
	/**
	 * Return the element from the pool without removing it.
	 * 
	 * @return the next available object to be polled from the queue.
	 */
	public E peek();
}