/*
 * MentaQueue => http://mentaqueue.soliveirajr.com
 * Copyright (C) 2012  Sergio Oliveira Jr. (sergio.oliveira.jr@gmail.com)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package org.mentaqueue;

import static org.mentaqueue.util.BuilderUtils.createBuilder;

import java.util.LinkedList;

import org.mentaqueue.util.Builder;

public class BlockingQueue<E> implements Queue<E> {

	private final static int DEFAULT_CAPACITY = 1026 * 16;

	private final LinkedList<E> queue;
	private final Object lock = new Object();
	private final int capacity;
	private final Builder<E> builder; 

	public BlockingQueue(int capacity, Builder<E> builder) {
		queue = new LinkedList<E>();
		this.capacity = capacity;
		this.builder = builder;
	}

	public BlockingQueue(Builder<E> builder) {
		this(DEFAULT_CAPACITY, builder);
	}
	
	public BlockingQueue(Class<E> klass) {
		this(createBuilder(klass));
	}
	
	public BlockingQueue(int capacity, Class<E> klass) {
		this(capacity, createBuilder(klass));
	}

	@Override
	public E nextToOffer() {
		return builder.newInstance();
	}

	@Override
	public void offer(E e) {
		synchronized(lock) {
			int size = queue.size();
			if (size == capacity) {
				try {
					lock.wait();
				} catch(Exception exc) {
					return;
				}
			}
			queue.add(e);
			if (size == 0) {
				lock.notify();
			}
		}
	}

	@Override
	public long available() {
		synchronized(lock) {
			if (queue.size() == 0) {
				try {
					lock.wait();
				} catch(Exception e) {
					return 0;
				}
			}
			return queue.size();
		}
	}

	@Override
	public E poll() {
		synchronized(lock) {
			E elem = queue.removeFirst();
			if (queue.size() == capacity - 1) {
				lock.notify();
			}
			return elem;
		}
	}

	@Override
	public void done() {
	}
}
