/*
 * MentaQueue => http://mentaqueue.soliveirajr.com
 * Copyright (C) 2012  Sergio Oliveira Jr. (sergio.oliveira.jr@gmail.com)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package org.mentaqueue;

import static org.mentaqueue.util.BuilderUtils.*;

import org.mentaqueue.util.Builder;
import org.mentaqueue.util.NumberUtils;
import org.mentaqueue.util.PaddedAtomicLong;

public class AtomicQueue<E> implements Queue<E> {

	private final static int DEFAULT_CAPACITY = 1024 * 16;

	private final int capacity;
	private final E[] data;
	private long nextOfferSequence = -1;
	private long nextPollSequence = -1;
	private long maxSequence;
	private final PaddedAtomicLong offerSequence = new PaddedAtomicLong(-1);
	private final PaddedAtomicLong pollSequence = new PaddedAtomicLong(-1);

	@SuppressWarnings("unchecked")
	public AtomicQueue(int capacity, Builder<E> builder) {
		NumberUtils.ensurePowerOfTwo(capacity);
		this.capacity = capacity;
		this.data = (E[]) new Object[capacity];
		for (int i = 0; i < capacity; i++) {
			this.data[i] = builder.newInstance();
		}
		this.maxSequence = findMaxSeqBeforeWrapping();
	}

	public AtomicQueue(Builder<E> builder) {
		this(DEFAULT_CAPACITY, builder);
	}
	
	public AtomicQueue(Class<E> klass) {
		this(createBuilder(klass));
	}
	
	public AtomicQueue(int capacity, Class<E> klass) {
		this(capacity, createBuilder(klass));
	}
	
	private final long findMaxSeqBeforeWrapping() {
		return capacity + pollSequence.get();
	}

	@Override
	public final E nextToOffer() {
		if (++nextOfferSequence > maxSequence) {
			// this would wrap the buffer... calculate the new one...
			this.maxSequence = findMaxSeqBeforeWrapping();
			if (nextOfferSequence > maxSequence) {
				nextOfferSequence--;
				return null;				
			}
		}
		return data[(int) (nextOfferSequence & capacity - 1)];
	}

	@Override
	public final void offer(E e) {
		offerSequence.lazySet(nextOfferSequence);
	}

	@Override
	public final long available() {
		return offerSequence.get() - nextPollSequence;
	}

	@Override
	public final E poll() {
		return data[(int) (++nextPollSequence & capacity - 1)];
	}

	@Override
	public final void done() {
		pollSequence.lazySet(nextPollSequence);
	}
}