/*
 * MentaQueue => http://mentaqueue.soliveirajr.com
 * Copyright (C) 2012  Sergio Oliveira Jr. (sergio.oliveira.jr@gmail.com)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package org.mentaqueue.test.sample;

import org.mentaqueue.AtomicQueue;
import org.mentaqueue.Queue;
import org.mentaqueue.wait.SpinYieldParkWaitStrategy;
import org.mentaqueue.wait.WaitStrategy;
import org.mentaqueue.wait.YieldParkWaitStrategy;

public class SampleCodeWithWaitStrategy {
	
	private static int MESSAGES = 1000 * 1000 * 10;

	public static class Entry {
		long sequence;
		StringBuilder message = new StringBuilder(256);
	}
	
	public static void main(String[] args) throws Exception {
		
		final WaitStrategy consumerWaitStrategy = new SpinYieldParkWaitStrategy(50, 50, true);
		final WaitStrategy producerWaitStrategy = new YieldParkWaitStrategy(10, true);
		
		final Queue<Entry> queue = new AtomicQueue<Entry>(1024 * 8, Entry.class);
		
		//// CONSUMER
		
		Thread t = new Thread() {
			
			@Override
			public void run() {
				boolean running = true;
				long expectedSequence = 1;
				while(running) {
					long x = queue.available();
					if (x > 0) {
						for(int i = 0; i < x; i++) {
							Entry entry = queue.poll();
							if (entry.sequence == expectedSequence) {
								// do whatever you want with entry.sequence and entry.message
								// you probably do not want to call toString() on entry.message
								// you can always print to system.out or write to a file one char at a time
								// without creating any garbage for the GC
								if (entry.sequence == MESSAGES) {
									running = false; // bye
								} else {
									expectedSequence++;
								}
							} else {
								throw new IllegalStateException("This should NEVER happen!");
							}
						}
						queue.done();
						consumerWaitStrategy.reset();
					} else {
						// we must wait, producer is lagging behind...
						consumerWaitStrategy.waitForOtherThread();
					}
				}
			}
		};
		
		long start = System.currentTimeMillis();
		
		t.start();
		
		//// PRODUCER
		
		for(int i = 1; i <= MESSAGES; i++) {
			Entry entry = queue.nextToOffer();
			if (entry == null) {
				// we must wait, consumer is lagging behind....
				producerWaitStrategy.waitForOtherThread();
				i--; // try again! (don't forget that!)
			} else {
				entry.sequence = i;
				entry.message.append("message_").append(entry.sequence);
				queue.offer(entry);
				producerWaitStrategy.reset();
			}
		}
		
		t.join();
		
		long time = System.currentTimeMillis() - start;
		
		System.out.println("DONE in " + time + " millis");
	}
}
