/*
 * MentaQueue => http://mentaqueue.soliveirajr.com
 * Copyright (C) 2012  Sergio Oliveira Jr. (sergio.oliveira.jr@gmail.com)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package org.mentaqueue.test.producer;

import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;

import org.mentaqueue.AtomicQueue;
import org.mentaqueue.BlockingLinkedQueue;
import org.mentaqueue.BrokenFastQueue;
import org.mentaqueue.ConcurrentArrayQueue;
import org.mentaqueue.ConcurrentLinkedQueue;
import org.mentaqueue.LockedQueue;
import org.mentaqueue.SynchronizedQueue;
import org.mentaqueue.VolatileQueue;
/**
 * <p>
 * To run: 
 * <pre>
 * java -Xms1g -Xmx4g -XX:NewSize=512m -XX:MaxNewSize=1024m -DtestNonBatchingQueues=true -cp target/classes org.mentaqueue.test.producer.TestAll 20 5 50 1000 1024
 * 
 * format: java -Xms1g -Xmx4g -XX:NewSize=512m -XX:MaxNewSize=1024m -DtestUncheckedQueues=true -DtestNonBatchingQueues=true -cp target/classes org.mentaqueue.test.producer.TestAll IGNORE OUTLIERS RUNS MESSAGES BUFFER_SIZE
 * IGNORE => How many initial runs will be ignored when calculating the average (warmup runs)?
 * OUTLIERS => How many best and worst to remove?
 * RUNS => How many times to run the test?
 * MESSAGES => How many messages to send to the other thread? (in thousands)
 * BUFFER_SIZE => How many messages can the buffer hold? (in multiples of 1024)
 * </pre>
 * </p>
 * <p>
 * Latest results:
 * <pre>
 * ******************************************************************
 *   LINUX
 * ******************************************************************
 * 
 * Intel(R) Xeon(R) CPU E5345 @ 2.33GHz
 * One physical processor with 6 logical processors
 * Java HotSpot(TM) 64-Bit Server VM (build 20.2-b06, mixed mode)
 * Linux soliveira-linux 2.6.32-30-generic x86_64 GNU/Linux
 * 
 * Final results:
 * 
 * 1: BrokenFastQueue => 1 nanoseconds/message
 * 2: AtomicQueue => 6 nanoseconds/message
 * 3: VolatileQueue => 12 nanoseconds/message (2.0x)
 * 4: AtomicQueue (*) => 51 nanoseconds/message (8.5x)
 * 5: VolatileQueue (*) => 71 nanoseconds/message (11.83x)
 * 6: SynchronizedQueue => 79 nanoseconds/message (13.17x)
 * 7: ConcurrentLinkedQueue (*) => 103 nanoseconds/message (17.17x)
 * 8: ConcurrentLinkedQueue => 105 nanoseconds/message (17.5x)
 * 9: LockedQueue => 111 nanoseconds/message (18.5x)
 * 10: BlockingLinkedQueue => 235 nanoseconds/message (39.17x)
 * 11: SynchronizedQueue (*) => 258 nanoseconds/message (43.0x)
 * 12: BlockingLinkedQueue (*) => 272 nanoseconds/message (45.33x)
 * 13: LockedQueue (*) => 274 nanoseconds/message (45.67x)
 * 14: ConcurrentArrayQueue => 403 nanoseconds/message (67.17x)
 * 15: ConcurrentArrayQueue (*) => 448 nanoseconds/message (74.67x)
 * 
 * (*) => Non-batching mode
 * 
 * Test details:
 * 
 * Warmup iterations per queue: 20
 * Outliers ignored (worst and best): 10
 * Total number of iterations: 50 (considered: 20)
 * Total number of messages sent to the other thread: 1,000,000
 * Queue buffer size: 1,048,576
 * Also test unchecked queues: false
 * Also test non-batching queues: true
 * 
 * </pre>
 * <pre>
 * ******************************************************************
 *   MAC OSX
 * ******************************************************************
 * 
 * MacBook Pro Intel Core 2 Duo 2.53 GHz
 * One physical processor with 2 logical processors
 * Java HotSpot(TM) 64-Bit Server VM (build 20.4-b02-402, mixed mode)
 * Darwin soliveira.local 10.8.0 Darwin Kernel Version 10.8.0 root:xnu-1504.15.3~1/RELEASE_I386 i386
 * 
 * Final results:
 * 
 * 1: BrokenFastQueue => 3 nanoseconds/message
 * 2: AtomicQueue => 15 nanoseconds/message
 * 3: AtomicQueue (*) => 23 nanoseconds/message (1.53x)
 * 4: VolatileQueue => 27 nanoseconds/message (1.8x)
 * 5: SynchronizedQueue => 36 nanoseconds/message (2.4x)
 * 6: VolatileQueue (*) => 41 nanoseconds/message (2.73x)
 * 7: LockedQueue => 56 nanoseconds/message (3.73x)
 * 8: SynchronizedQueue (*) => 118 nanoseconds/message (7.87x)
 * 9: ConcurrentLinkedQueue (*) => 150 nanoseconds/message (10.0x)
 * 10: ConcurrentLinkedQueue => 170 nanoseconds/message (11.33x)
 * 11: LockedQueue (*) => 174 nanoseconds/message (11.6x)
 * 12: BlockingLinkedQueue (*) => 278 nanoseconds/message (18.53x)
 * 13: ConcurrentArrayQueue => 294 nanoseconds/message (19.6x)
 * 14: BlockingLinkedQueue => 324 nanoseconds/message (21.6x)
 * 15: ConcurrentArrayQueue (*) => 363 nanoseconds/message (24.2x)
 * 
 * (*) => Non-batching mode
 * 
 * Test details:
 * 
 * Warmup iterations per queue: 20
 * Outliers ignored (worst and best): 10
 * Total number of iterations: 50 (considered: 20)
 * Total number of messages sent to the other thread: 1,000,000
 * Queue buffer size: 1,048,576
 * Also test unchecked queues: false
 * Also test non-batching queues: true 
 * </pre>
 * </p>
 * @author Sergio Oliveira Jr.
 */
public class TestAll {

	private static final Map<Long, String> results = new TreeMap<Long, String>();
	
	private static final NumberFormat NUMBER_FORMATTER = new DecimalFormat("#,###,###");

	// We don't want to overwrite results with same value!
	private static final void addResult(long value, String res) {
		while(results.containsKey(value)) {
			value--; // make it worse to be fair...
		}
		results.put(value, res);
	}
	
	public static void main(String[] args) throws Exception {
		
		if (args.length != 5) {
			System.out.println("format: java -Xms1g -Xmx4g -XX:NewSize=512m -XX:MaxNewSize=1024m -DtestUncheckedQueues=true -DtestNonBatchingQueues=true -cp target/classes org.mentaqueue.test.TestAll IGNORE OUTLIERS RUNS MESSAGES BUFFER_SIZE");
			System.out.println("IGNORE => How many initial runs will be ignored when calculating the average (warmup runs)?");
			System.out.println("OUTLIERS => How many best and worst to remove?");
			System.out.println("RUNS => How many times to run the test?");
			System.out.println("MESSAGES => How many messages to send to the other thread? (in thousands)");
			System.out.println("BUFFER_SIZE => How many messages can the buffer hold? (in multiples of 1024)");
			System.out.println();
			return;
		}
		
		String[] params = new String[6];
		for(int i = 0; i < args.length; i++) {
			params[i] = args[i];
		}
		params[5] = "true";

		String s = System.getProperty("testNonBatchingQueues");
		boolean testNonBatching = false;
		if (s != null && s.equals("true")) {
			testNonBatching = true;
		}
		
		for(int i = 0; i < 2; i++) {
			
			if (i == 1) {
				if (!testNonBatching) {
					break;
				} else {
					System.out.println();
					System.out.println("Now testing queues on non-batching mode:");
					System.out.println();
				}
			}
			
			String suffix = "";
			
			if (i == 1 && testNonBatching) {
				suffix = " (*)";
				params[5] = "false";
			}
		
			if (i == 0) {
				TestBrokenFastQueue.main(params);
				addResult(TestBrokenFastQueue.getResult(), BrokenFastQueue.class.getSimpleName() + suffix);
			}
			
			TestAtomicQueue.main(params);
			addResult(TestAtomicQueue.getResult(), AtomicQueue.class.getSimpleName() + suffix);
			
			TestVolatileQueue.main(params);
			addResult(TestVolatileQueue.getResult(), VolatileQueue.class.getSimpleName() + suffix);
			
			TestLockedQueue.main(params);
			addResult(TestLockedQueue.getResult(), LockedQueue.class.getSimpleName() + suffix);
			
			TestSynchronizedQueue.main(params);
			addResult(TestSynchronizedQueue.getResult(), SynchronizedQueue.class.getSimpleName() + suffix);
			
			TestConcurrentLinkedQueue.main(params);
			addResult(TestConcurrentLinkedQueue.getResult(), ConcurrentLinkedQueue.class.getSimpleName() + suffix);
	
			TestConcurrentArrayQueue.main(params);
			addResult(TestConcurrentArrayQueue.getResult(), ConcurrentArrayQueue.class.getSimpleName() + suffix);
			
			TestBlockingLinkedQueue.main(params);
			addResult(TestBlockingLinkedQueue.getResult(), BlockingLinkedQueue.class.getSimpleName() + suffix);
		}
		
		System.out.println();
		System.out.println("Final results:\n");
		
		Iterator<Long> iter = results.keySet().iterator();
		
		int count = 1;
		
		long champion = 0;
		
		while(iter.hasNext()) {
			long nanosPerMsg = iter.next();
			String queue = results.get(nanosPerMsg);
			String res = count + ": " + queue + " => " + NUMBER_FORMATTER.format(nanosPerMsg) + " nanoseconds/message";
			
			if (count == 2) {
				// first one must be BrokenFastQueue
				champion = nanosPerMsg;
			} else if (count > 2) {
				res += " (" + roundTwoDecimals(nanosPerMsg, champion) + "x)";
			}
			
			System.out.println(res);
			
			count++;
		}
		
		System.out.println();
		if (testNonBatching) {
			System.out.println("(*) => Non-batching mode");
			System.out.println();
		}
		
		System.out.println("Test details:");
		System.out.println();
		
		System.out.println("Warmup iterations per queue: " + args[0]);
		System.out.println("Outliers ignored (worst and best): " + Integer.parseInt(args[1]) * 2);
		
		int considered = Integer.parseInt(args[2]) - (Integer.parseInt(args[0]) + Integer.parseInt(args[1]) * 2);
		
		System.out.println("Total number of iterations: " + args[2] + " (considered: " + considered + ")");
		System.out.println("Total number of messages sent to the other thread: " + NUMBER_FORMATTER.format(Integer.parseInt(args[3]) * 1000));
		System.out.println("Queue buffer size: " + NUMBER_FORMATTER.format(Integer.parseInt(args[4]) * 1024));
		System.out.println("Also test non-batching queues: " + testNonBatching);
		System.out.println();
	}
	
	private static String roundTwoDecimals(long x, long y) {
		float f = (float) x / (float) y;
		f *= 100f;
		f = Math.round(f);
		f /= 100f;
		return String.valueOf(f);
	}
}
