/*
 * MentaQueue => http://mentaqueue.soliveirajr.com
 * Copyright (C) 2012  Sergio Oliveira Jr. (sergio.oliveira.jr@gmail.com)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package org.mentaqueue.test.messages;

import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;

import org.mentaqueue.AtomicQueue;
import org.mentaqueue.BlockingLinkedQueue;
import org.mentaqueue.BrokenFastQueue;
import org.mentaqueue.ConcurrentArrayQueue;
import org.mentaqueue.ConcurrentLinkedQueue;
import org.mentaqueue.LockedQueue;
import org.mentaqueue.SynchronizedQueue;
import org.mentaqueue.VolatileQueue;
/**
 * <p>
 * To run: 
 * <pre>
 * java -Xms1g -Xmx4g -XX:NewSize=512m -XX:MaxNewSize=1024m -DtestNonBatchingQueues=true -cp target/classes org.mentaqueue.test.messages.TestAll 20 5 50 1000 1024
 * 
 * format: java -Xms1g -Xmx4g -XX:NewSize=512m -XX:MaxNewSize=1024m -DtestUncheckedQueues=true -DtestNonBatchingQueues=true -cp target/classes org.mentaqueue.test.messages.TestAll IGNORE OUTLIERS RUNS MESSAGES BUFFER_SIZE
 * IGNORE => How many initial runs will be ignored when calculating the average (warmup runs)?
 * OUTLIERS => How many best and worst to remove?
 * RUNS => How many times to run the test?
 * MESSAGES => How many messages to send to the other thread? (in thousands)
 * BUFFER_SIZE => How many messages can the buffer hold? (in multiples of 1024)
 * </pre>
 * </p>
 * <p>
 * Latest results:
 * <pre>
 * ******************************************************************
 *   LINUX
 * ******************************************************************
 * 
 * Intel(R) Xeon(R) CPU E5345 @ 2.33GHz
 * One physical processor with 6 logical processors
 * Java HotSpot(TM) 64-Bit Server VM (build 20.2-b06, mixed mode)
 * Linux soliveira-linux 2.6.32-30-generic x86_64 GNU/Linux
 * 
 * Final results:
 * 
 * 1: BrokenFastQueue => 550,000,000 messages/sec
 * 2: AtomicQueue => 125,892,857 messages/sec
 * 3: VolatileQueue => 73,901,098 messages/sec (1.7x)
 * 4: SynchronizedQueue => 47,294,371 messages/sec (2.66x)
 * 5: LockedQueue => 20,779,463 messages/sec (6.06x)
 * 6: AtomicQueue (*) => 19,067,488 messages/sec (6.6x)
 * 7: VolatileQueue (*) => 14,705,882 messages/sec (8.56x)
 * 8: ConcurrentLinkedQueue => 7,908,385 messages/sec (15.92x)
 * 9: ConcurrentLinkedQueue (*) => 7,355,664 messages/sec (17.12x)
 * 10: SynchronizedQueue (*) => 5,606,576 messages/sec (22.45x)
 * 11: BlockingLinkedQueue => 3,751,337 messages/sec (33.56x)
 * 12: LockedQueue (*) => 3,261,586 messages/sec (38.6x)
 * 13: BlockingLinkedQueue (*) => 3,169,477 messages/sec (39.72x)
 * 14: ConcurrentArrayQueue => 2,469,454 messages/sec (50.98x)
 * 15: ConcurrentArrayQueue (*) => 2,056,815 messages/sec (61.21x)
 * 
 * (*) => Non-batching mode
 * 
 * Test details:
 * 
 * Warmup iterations per queue: 20
 * Outliers ignored (worst and best): 10
 * Total number of iterations: 50 (considered: 20)
 * Total number of messages sent to the other thread: 1,000,000
 * Queue buffer size: 1,048,576
 * Also test unchecked queues: false
 * Also test non-batching queues: true
 * </pre>
 * <pre>
 * ******************************************************************
 *   MAC OSX
 * ******************************************************************
 * 
 * MacBook Pro Intel Core 2 Duo 2.53 GHz
 * One physical processor with 2 logical processors
 * Java HotSpot(TM) 64-Bit Server VM (build 20.4-b02-402, mixed mode)
 * Darwin soliveira.local 10.8.0 Darwin Kernel Version 10.8.0 root:xnu-1504.15.3~1/RELEASE_I386 i386
 * 
 * 1: BrokenFastQueue => 408,333,333 messages/sec
 * 2: <b>AtomicQueue => 52,105,262 messages/sec</b>
 * 3: VolatileQueue => 29,327,730 messages/sec (1.78x)
 * 4: SynchronizedQueue => 24,825,013 messages/sec (2.1x)
 * 5: AtomicQueue (*) => 24,779,912 messages/sec (2.1x)
 * 6: VolatileQueue (*) => 18,675,750 messages/sec (2.79x)
 * 7: LockedQueue => 17,051,431 messages/sec (3.06x)
 * 8: ConcurrentLinkedQueue (*) => 6,686,979 messages/sec (7.79x)
 * 9: ConcurrentLinkedQueue => 6,351,515 messages/sec (8.2x)
 * 10: SynchronizedQueue (*) => 5,335,418 messages/sec (9.77x)
 * 11: LockedQueue (*) => 4,251,766 messages/sec (12.25x)
 * 12: ConcurrentArrayQueue => 3,205,216 messages/sec (16.26x)
 * 13: BlockingLinkedQueue (*) => 3,101,286 messages/sec (16.8x)
 * 14: BlockingLinkedQueue => 2,774,668 messages/sec (18.78x)
 * 15: ConcurrentArrayQueue (*) => 2,261,947 messages/sec (23.04x)
 * 
 * (*) => Non-batching mode
 * 
 * Test details: 
 * 
 * Warmup iterations per queue: 20
 * Outliers ignored (worst and best): 10
 * Total number of iterations: 50 (considered: 20)
 * Total number of messages sent to the other thread: 1,000,000
 * Queue buffer size: 131,072
 * Also test unchecked queues: false
 * Also test non-batching queues: true
 * </pre>
 * </p>
 * @author Sergio Oliveira Jr.
 */
public class TestAll {

	private static final Map<Long, String> results = new TreeMap<Long, String>(Collections.reverseOrder());
	
	private static final NumberFormat NUMBER_FORMATTER = new DecimalFormat("#,###,###");

	// We don't want to overwrite results with same value!
	private static final void addResult(long value, String res) {
		while(results.containsKey(value)) {
			value--; // make it worse to be fair...
		}
		results.put(value, res);
	}
	
	public static void main(String[] args) throws Exception {
		
		if (args.length != 5) {
			System.out.println("format: java -Xms1g -Xmx4g -XX:NewSize=512m -XX:MaxNewSize=1024m -DtestUncheckedQueues=true -DtestNonBatchingQueues=true -cp target/classes org.mentaqueue.test.TestAll IGNORE OUTLIERS RUNS MESSAGES BUFFER_SIZE");
			System.out.println("IGNORE => How many initial runs will be ignored when calculating the average (warmup runs)?");
			System.out.println("OUTLIERS => How many best and worst to remove?");
			System.out.println("RUNS => How many times to run the test?");
			System.out.println("MESSAGES => How many messages to send to the other thread? (in thousands)");
			System.out.println("BUFFER_SIZE => How many messages can the buffer hold? (in multiples of 1024)");
			System.out.println();
			return;
		}
		
		String[] params = new String[6];
		for(int i = 0; i < args.length; i++) {
			params[i] = args[i];
		}
		params[5] = "true";

		String s = System.getProperty("testNonBatchingQueues");
		boolean testNonBatching = false;
		if (s != null && s.equals("true")) {
			testNonBatching = true;
		}
		
		for(int i = 0; i < 2; i++) {
			
			if (i == 1) {
				if (!testNonBatching) {
					break;
				} else {
					System.out.println();
					System.out.println("Now testing queues on non-batching mode:");
					System.out.println();
				}
			}
			
			String suffix = "";
			
			if (i == 1 && testNonBatching) {
				suffix = " (*)";
				params[5] = "false";
			}
		
			if (i == 0) {
				TestBrokenFastQueue.main(params);
				addResult(TestBrokenFastQueue.getResult(), BrokenFastQueue.class.getSimpleName() + suffix);
			}
			
			TestAtomicQueue.main(params);
			addResult(TestAtomicQueue.getResult(), AtomicQueue.class.getSimpleName() + suffix);
			
			TestVolatileQueue.main(params);
			addResult(TestVolatileQueue.getResult(), VolatileQueue.class.getSimpleName() + suffix);
			
			TestLockedQueue.main(params);
			addResult(TestLockedQueue.getResult(), LockedQueue.class.getSimpleName() + suffix);
			
			TestSynchronizedQueue.main(params);
			addResult(TestSynchronizedQueue.getResult(), SynchronizedQueue.class.getSimpleName() + suffix);
			
			TestConcurrentLinkedQueue.main(params);
			addResult(TestConcurrentLinkedQueue.getResult(), ConcurrentLinkedQueue.class.getSimpleName() + suffix);
	
			TestConcurrentArrayQueue.main(params);
			addResult(TestConcurrentArrayQueue.getResult(), ConcurrentArrayQueue.class.getSimpleName() + suffix);
			
			TestBlockingLinkedQueue.main(params);
			addResult(TestBlockingLinkedQueue.getResult(), BlockingLinkedQueue.class.getSimpleName() + suffix);
		}
		
		System.out.println();
		System.out.println("Final results:\n");
		
		Iterator<Long> iter = results.keySet().iterator();
		
		int count = 1;
		
		long champion = 0;
		
		while(iter.hasNext()) {
			long messagesPerSec = iter.next();
			String queue = results.get(messagesPerSec);
			String res = count + ": " + queue + " => " + NUMBER_FORMATTER.format(messagesPerSec) + " messages/sec";
			
			if (count == 2) {
				// first one must be BrokenFastQueue
				champion = messagesPerSec;
			} else if (count > 2) {
				res += " (" + roundTwoDecimals(champion, messagesPerSec) + "x)";
			}
			
			System.out.println(res);
			
			count++;
		}
		
		System.out.println();
		if (testNonBatching) {
			System.out.println("(*) => Non-batching mode");
			System.out.println();
		}
		
		System.out.println("Test details:");
		System.out.println();
		
		System.out.println("Warmup iterations per queue: " + args[0]);
		System.out.println("Outliers ignored (worst and best): " + Integer.parseInt(args[1]) * 2);
		
		int considered = Integer.parseInt(args[2]) - (Integer.parseInt(args[0]) + Integer.parseInt(args[1]) * 2);
		
		System.out.println("Total number of iterations: " + args[2] + " (considered: " + considered + ")");
		System.out.println("Total number of messages sent to the other thread: " + NUMBER_FORMATTER.format(Integer.parseInt(args[3]) * 1000));
		System.out.println("Queue buffer size: " + NUMBER_FORMATTER.format(Integer.parseInt(args[4]) * 1024));
		System.out.println("Also test non-batching queues: " + testNonBatching);
		System.out.println();
	}
	
	private static String roundTwoDecimals(long x, long y) {
		float f = (float) x / (float) y;
		f *= 100f;
		f = Math.round(f);
		f /= 100f;
		return String.valueOf(f);
	}
}
