/*
 * MentaQueue => http://mentaqueue.soliveirajr.com
 * Copyright (C) 2012  Sergio Oliveira Jr. (sergio.oliveira.jr@gmail.com)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package org.mentaqueue;

import org.mentaqueue.util.Builder;
import org.mentaqueue.util.NumberUtils;

public class SynchronizedQueue<E> implements Queue<E> {

	private final static int DEFAULT_CAPACITY = 1024 * 16;

	private final int capacity;
	private final E[] data;
	private long nextOfferValue = -1;
	private long nextPollValue = -1;
	private long offeringSequence = -1;
	private long pollingSequence = -1;
	private final Object offeringLock = new Object();
	private final Object pollingLock = new Object();
	private long maxSeq;

	@SuppressWarnings("unchecked")
	public SynchronizedQueue(int capacity, Builder<E> builder) {
		NumberUtils.ensurePowerOfTwo(capacity);
		this.capacity = capacity;
		this.data = (E[]) new Object[capacity];
		for (int i = 0; i < capacity; i++) {
			this.data[i] = builder.newInstance();
		}
		this.maxSeq = findMaxSeqBeforeWrapping();
	}

	public SynchronizedQueue(Builder<E> builder) {
		this(DEFAULT_CAPACITY, builder);
	}
	
	private final static <E> Builder<E> createBuilder(final Class<E> klass) {
		return new Builder<E>() {
			@Override
            public E newInstance() {
	            try {
	            	return klass.newInstance();
	            } catch(Exception e) {
	            	throw new RuntimeException(e);
	            }
            }
		};
	}
	
	public SynchronizedQueue(Class<E> klass) {
		this(createBuilder(klass));
	}
	
	public SynchronizedQueue(int capacity, Class<E> klass) {
		this(capacity, createBuilder(klass));
	}

	
	private final long findMaxSeqBeforeWrapping() {
		synchronized(pollingLock) {
			return capacity + pollingSequence;
		}
	}

	@Override
	public E nextToOffer() {
		long index = ++nextOfferValue;
		if (index <= maxSeq) {
			// we are safe
		} else {
			// recalculate max sequence...
			this.maxSeq = findMaxSeqBeforeWrapping();
			if (index > maxSeq) {
				// we are still ahead... nothing to do for now...
				nextOfferValue--;
				return null;
			}
		}
		return data[(int) (index & capacity - 1)];
	}

	@Override
	public void offer(E e) {
		synchronized (offeringLock) {
			offeringSequence = nextOfferValue;
		}
	}

	@Override
	public long available() {
		synchronized (offeringLock) {
			return offeringSequence - nextPollValue;
		}
	}

	@Override
	public E poll() {
		return data[(int) (++nextPollValue & capacity - 1)];
	}

	@Override
	public void done() {
		synchronized (pollingLock) {
			pollingSequence = nextPollValue;
		}
	}
}