/*
 * MentaQueue => http://mentaqueue.soliveirajr.com
 * Copyright (C) 2012  Sergio Oliveira Jr. (sergio.oliveira.jr@gmail.com)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package org.mentaqueue;

/**
 * A queue API that allows batching and pooling of objects.
 * 
 *  So to offer you first get a mutable object from the queue by calling <i>nextToOffer()</i>, alter the object and call <i>offer()</i>.
 *  That allows the queue objects to be pooled, avoiding any garbage collection.
 *  
 *  And to poll you first call <i>available()</i> to know how many objects you can safely poll, call <i>poll()</i> in a loop and when done call <i>done()</i>.
 *  That allows polling to be batched, so you pay a synchronization price only when you call <i>available()</i> and NOT when you call <i>poll()</i>.
 * 
 * @author Sergio Oliveira Jr.
 *
 * @param <E>
 */
public interface Queue<E> {

	/**
	 * Return the next pooled mutable object that can be used by the producer.
	 * 
	 * @return the next mutable object that can be used by the producer.
	 */
	public E nextToOffer();

	/**
	 * Offer an object to the queue. The object must have been previously obtained by calling <i>nextToOffer()</i>.
	 * 
	 * @param e the object to be offered to the queue.
	 */
	public void offer(E e);

	/**
	 * Return the number of objects that can be safely polled from this queue. It can return zero.
	 * 
	 * @return number of objects that can be polled.
	 */
	public long available();

	/**
	 * Poll a object from the queue. You can only call this method after calling <i>available()</i> so you
	 * know what is the maximum times you can call it.
	 * 
	 * NOTE: You should NOT keep your own reference for this mutable object. Read what you need to get from it and release its reference.
	 * 
	 * @return an object from the queue.
	 */
	public E poll();

	/**
	 * Called to indicate that all polling have been done.
	 */
	public void done();
}