/*
 * MentaQueue => http://mentaqueue.soliveirajr.com
 * Copyright (C) 2012  Sergio Oliveira Jr. (sergio.oliveira.jr@gmail.com)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package org.mentaqueue;

import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.mentaqueue.util.Builder;
import org.mentaqueue.util.NumberUtils;

public class LockedQueue<E> implements Queue<E> {

	private final static int DEFAULT_CAPACITY = 1024 * 16;

	private final int capacity;
	private final E[] data;
	private long nextOfferValue = -1;
	private long nextPollValue = -1;
	private long offeringSequence = -1;
	private long pollingSequence = -1;
	private final Lock offeringLock = new ReentrantLock();
	private final Lock pollingLock = new ReentrantLock();
	private long maxSeq;

	@SuppressWarnings("unchecked")
	public LockedQueue(int capacity, Builder<E> builder) {
		NumberUtils.ensurePowerOfTwo(capacity);
		this.capacity = capacity;
		this.data = (E[]) new Object[capacity];
		for (int i = 0; i < capacity; i++) {
			this.data[i] = builder.newInstance();
		}
		this.maxSeq = findMaxSeqBeforeWrapping();
	}

	public LockedQueue(Builder<E> builder) {
		this(DEFAULT_CAPACITY, builder);
	}
	
	private final static <E> Builder<E> createBuilder(final Class<E> klass) {
		return new Builder<E>() {
			@Override
            public E newInstance() {
	            try {
	            	return klass.newInstance();
	            } catch(Exception e) {
	            	throw new RuntimeException(e);
	            }
            }
		};
	}
	
	public LockedQueue(Class<E> klass) {
		this(createBuilder(klass));
	}
	
	public LockedQueue(int capacity, Class<E> klass) {
		this(capacity, createBuilder(klass));
	}

	
	private final long findMaxSeqBeforeWrapping() {
		pollingLock.lock();
		try {
			return capacity + pollingSequence;
		} finally {
			pollingLock.unlock();
		}
	}

	@Override
	public E nextToOffer() {
		long index = ++nextOfferValue;
		if (index <= maxSeq) {
			// we are safe
		} else {
			// recalculate max sequence...
			this.maxSeq = findMaxSeqBeforeWrapping();
			if (index > maxSeq) {
				// we are still ahead... nothing to do for now...
				nextOfferValue--;
				return null;
			}
		}
		return data[(int) (index & capacity - 1)];
	}

	@Override
	public void offer(E e) {
		offeringLock.lock();
		offeringSequence = nextOfferValue;
		offeringLock.unlock();
	}

	@Override
	public long available() {
		offeringLock.lock();
		try {
			return offeringSequence - nextPollValue;
		} finally {
			offeringLock.unlock();
		}
	}

	@Override
	public E poll() {
		return data[(int) (++nextPollValue & capacity - 1)];
	}

	@Override
	public void done() {
		pollingLock.lock();
		pollingSequence = nextPollValue;
		pollingLock.unlock();
	}
}