package  org.mentabean.util;

import java.lang.reflect.Method;
import java.util.Date;

import javassist.util.proxy.MethodFilter;
import javassist.util.proxy.MethodHandler;
import javassist.util.proxy.ProxyFactory;

import org.mentabean.BeanException;

public class PropertiesProxy {
	
	private static final ThreadLocal<String> propertyNames = new ThreadLocal<String>();
	
	public static String get() {
		String propName = propertyNames.get();
		if (propName == null) {
			throw new BeanException("Was not able to get property name through the proxy!");
		}
		return propName;
	}
	
	/**
	 * Return the property name, if the method is a valid JavaBean getter
	 * 
	 * @param method the method
	 * @return the property name of null if not a valid getter
	 */
	private static String getPropName(Method method) {
		
		String methodName = method.getName();
		Class<?> propType = method.getReturnType();
		
		if (propType.equals(Void.class)) return null; // not a getter
		
		Class<?>[] params = method.getParameterTypes();
		if (params != null && params.length > 0) return null; // not a getter
		
		String propName;
		
		if (methodName.startsWith("get") && methodName.length() > 3) {
		
			propName = methodName.substring(3);
			
		} else if (methodName.startsWith("is") && methodName.length() > 2 && (propType.equals(boolean.class) || propType.equals(Boolean.class))) {
			
			propName = methodName.substring(2);
			
		} else {
			
			return null; // not a getter...
		}
		
		propName = propName.substring(0, 1).toLowerCase() + propName.substring(1); // first letter is lower-case
		
		if (propName.equals("class")) return null; // not a property...
		
		return propName;
	}
	
	public static <E> E create(Class<E> klass) {
		
		try {
		
    		ProxyFactory factory = new ProxyFactory();
    		factory.setSuperclass(klass);
    		
    		factory.setFilter(new MethodFilter() {
    
    			@Override
                public boolean isHandled(Method m) {
    				return getPropName(m) != null;
    			}
    		});
    		
    		MethodHandler handler = new MethodHandler() {
    	        
    			@Override
    	        public Object invoke(Object self, Method thisMethod, Method proceed, Object[] args) throws Throwable {
    				
    				propertyNames.set(getPropName(thisMethod));
    				
    				Class<?> propType = thisMethod.getReturnType();
    				
    				// take care of primitives that cannot be null...
    				
    				if (propType.equals(boolean.class)) {
    					return false;
    				} else if (propType.equals(char.class)) {
    					return (char) 0;
    				} else if (propType.equals(byte.class)) {
    					return (byte) 0;
    				} else if (propType.equals(long.class)) {
    					return (long) 0;
    				} else if (propType.equals(int.class)) {
    					return (int) 0;
    				} else if (propType.equals(short.class)) {
    					return (short) 0;
    				} else if (propType.equals(float.class)) {
    					return (float) 0;
    				} else if (propType.equals(double.class)) {
    					return (double) 0;
    				} else {
    					return null;
    				}
    	        }
    	    };
    	    
    	    return (E) factory.create(new Class<?>[0], new Object[0], handler);
    	    
		} catch(Exception e) {
			throw new BeanException(e);
		}
	}
	
    public static void main(String[] args) throws Exception {
    	
    	// first try the hardcoded way:
    	field("age");
    	
    	System.out.println();
    	
    	// now try the new way:

    	User user = PropertiesProxy.create(User.class);
    	
    	field(user.getAge());
    	field(user.getUsername());
    	field(user.isBlah1());
    	field(user.getBlah2());
    	field(user.getBlah3());
    	field(user.getBlah4());
    	field(user.getBlah5());
    	field(user.getBlah6());
    }
    
    public static void field(String propName) {
    	if (propName == null) { // this is necessary for when the property is a String...
    		field((Object) null);
    	} else {
    		System.out.println("hardcoded propName: " + propName);
    	}
    }
    
    public static void field(Object propName) {
    	propName = PropertiesProxy.get();
    	System.out.println("proxified propName: " + propName);
    }
    
    public static class User {
    	
    	private int age;
    	private String username;
    	private boolean blah1;
    	private Boolean blah2;
    	private char blah3;
    	private long blah4;
    	private Long blah5;
    	private Date blah6;
    	
    	public User() { 
    		
    	}

		public int getAge() {
			return age;
		}

		public void setAge(int age) {
			this.age = age;
		}

		public String getUsername() {
			return username;
		}

		public void setUsername(String username) {
			this.username = username;
		}

		public boolean isBlah1() {
			return blah1;
		}

		public void setBlah1(boolean blah1) {
			this.blah1 = blah1;
		}

		public Boolean getBlah2() {
			return blah2;
		}

		public void setBlah2(Boolean blah2) {
			this.blah2 = blah2;
		}

		public char getBlah3() {
			return blah3;
		}

		public void setBlah3(char blah3) {
			this.blah3 = blah3;
		}

		public long getBlah4() {
			return blah4;
		}

		public void setBlah4(long blah4) {
			this.blah4 = blah4;
		}

		public Long getBlah5() {
			return blah5;
		}

		public void setBlah5(Long blah5) {
			this.blah5 = blah5;
		}

		public Date getBlah6() {
			return blah6;
		}

		public void setBlah6(Date blah6) {
			this.blah6 = blah6;
		}
    }
}