/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * MentaBean => http://www.mentabean.org
 * Author: Sergio Oliveira Jr. (sergio.oliveira.jr@gmail.com)
 */
package org.mentabean;

import java.sql.Connection;
import java.sql.ResultSet;
import java.util.List;

/**
 * Describe a simple ORM interface that can perform CRUD for Beans according to properties defined programmatically on BeanManager. It can also load lists and unique beans based on properties set on a given bean. It also supports dynamic update, in other words, it will only update fields from a bean
 * loaded from the database that were actually modified during the session. The behavior can be turned off when necessary ,in other words, when you want to blindly update all properties from a bean in the database regarding of whether they were modified or not.
 * 
 * @author Sergio Oliveira Jr.
 */
public interface BeanSession {

	/**
	 * Get the database connection being used by this bean session.
	 * 
	 * @return the database connection
	 */
	public Connection getConnection();

	/**
	 * Load the bean from the database, injecting all its properties through reflection. Note that the bean passed MUST have its primary key set otherwise there is no way we can load it from the database.
	 * 
	 * @param bean
	 *            The bean we want to load from the DB.
	 * @return true if the bean was found in the database, false otherwise
	 */
	public boolean load(Object bean);

	/**
	 * Update the bean in the database. Only the bean fields that have been modified (dirty) will be updated.
	 * 
	 * It will return 1 if an update did happen or 0 if the bean could not be found in the database or if there was nothing modified in bean.
	 * 
	 * The bean MUST have its primary key set, otherwise it is impossible to update the bean in the database, and an exception will be thrown.
	 * 
	 * @param bean
	 *            The bean to be updated
	 * @return 1 if update was successful, 0 if the update did not happen or was not necessary
	 */
	public int update(Object bean);

	/**
	 * Same as update(bean) but here you can turn off the default dynamic update behavior and force all bean properties to be updated regardless whether they have been modified or not.
	 * 
	 * @param bean
	 * @return the number of rows that were updated
	 * @throws Exception
	 */
	public int updateAll(Object bean);

	/**
	 * Insert the bean in the database.
	 * 
	 * Depending on the type of PK, the generation of the PK can and should be taken care by the DB itself. The generated PK should be inserted in the bean by reflection, before the method returns.
	 * 
	 * The default, database-independent implementation of this method, insert all fields in the database not worrying about PK generation strategies.
	 * 
	 * @param bean
	 *            The bean to insert
	 */
	public void insert(Object bean);

	/**
	 * Delete the bean from the database.
	 * 
	 * The PK of the bean MUST be set. The bean can only be deleted by its PK.
	 * 
	 * @param bean
	 * @return true if it was deleted or false if it did not exist
	 * @throws Exception
	 */
	public boolean delete(Object bean);

	/**
	 * Count the number of beans that would be returned by a loadList method.
	 * 
	 * @param bean
	 *            The bean holding the properties used by the list query.
	 * @return the number of beans found in the database
	 */
	public int countList(Object bean);

	/**
	 * Load a list of beans based on the properties present in the bean passed. For example, if you want to load all users with lastName equal to 'saoj' you would instantiate a bean and set its lastName property to 'saoj' before passing it as an argument to this method.
	 * 
	 * @param <E>
	 * @param bean
	 *            The bean holding the properties used by the list query.
	 * @return A list of beans the match the properties in the given bean. Or an empty list if nothing was found.
	 */
	public <E> List<E> loadList(E bean);

	public <E> List<E> loadList(E bean, String[] properties);

	public <E> List<E> loadListMinus(E bean, String[] minus);

	/**
	 * Same as loadList(bean) except that you can order the list returned by passing an SQL orderBy clause.
	 * 
	 * @param <E>
	 * @param bean
	 *            The bean holding the properties used by the list query.
	 * @param orderBy
	 *            The orderBy SQL clause.
	 * @return A list of beans the match the properties in the given bean. Or an empty list if nothing was found.
	 */
	public <E> List<E> loadList(E bean, String orderBy);

	public <E> List<E> loadList(E bean, String orderBy, String[] properties);

	public <E> List<E> loadListMinus(E bean, String orderBy, String[] minus);

	/**
	 * Same as loadList(bean) except that you can limit the number of beans returned in the list.
	 * 
	 * @param <E>
	 * @param bean
	 *            The bean holding the properties used by the list query.
	 * @param limit
	 *            The max number of beans returned in the list.
	 * @return A list of beans the match the properties in the given bean. Or an empty list if nothing was found.
	 */
	public <E> List<E> loadList(E bean, int limit);

	public <E> List<E> loadList(E bean, int limit, String[] properties);

	public <E> List<E> loadListMinus(E bean, int limit, String[] minus);

	/**
	 * Same as loadList(bean) except that you can limit the number of beans returned in the list as well as sort them by passing a orderBy SQL clause.
	 * 
	 * @param <E>
	 * @param bean
	 *            The bean holding the properties used by the list query.
	 * @param orderBy
	 *            The orderBy SQL clause.
	 * @param limit
	 *            The max number of beans returned in the list.
	 * @return A list of beans the match the properties in the given bean. Or an empty list if nothing was found.
	 */
	public <E> List<E> loadList(E bean, String orderBy, int limit);

	public <E> List<E> loadList(E bean, String orderBy, int limit, String[] properties);

	public <E> List<E> loadListMinus(E bean, String orderBy, int limit, String[] minus);

	/**
	 * Same as loadList(bean) but it attempts to load a single bean only. If more than one bean is found it throws an exception.
	 * 
	 * @param <E>
	 * @param bean
	 *            The bean holding the properties used by the load query.
	 * @return A unique bean that match the properties in the given bean. Or null if nothing was found.
	 * @throws BeanException
	 *             if more than one bean is found by the query.
	 */
	public <E> E loadUnique(E bean);

	public String buildSelect(final Class<? extends Object> beanClass);

	public String buildSelect(final Class<? extends Object> beanClass, String[] properties);

	public String buildSelectMinus(final Class<? extends Object> beanClass, String[] minus);

	public String buildSelect(final Class<? extends Object> beanClass, String tableName);

	public String buildSelect(final Class<? extends Object> beanClass, String tableName, String[] properties);

	public String buildSelectMinus(final Class<? extends Object> beanClass, String tableName, String[] minus);

	public void populateBean(final ResultSet rset, final Object bean);

	public void populateBean(final ResultSet rset, final Object bean, String[] properties);

	public void populateBeanMinus(final ResultSet rset, final Object bean, String[] minus);

	public void populateBean(final ResultSet rset, final Object bean, String tableName);

	public void populateBean(final ResultSet rset, final Object bean, String tableName, String[] properties);

	public void populateBeanMinus(final ResultSet rset, final Object bean, String tableName, String[] minus);

	// some experiment with metadata...

	public void createTable(Class<? extends Object> beanKlass);

	public void createTables();

}