/*
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * MentaBean => http://www.mentabean.org
 * Author: Sergio Oliveira Jr. (sergio.oliveira.jr@gmail.com)
 */
package org.mentabean.type;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import org.mentabean.DBType;

public class BooleanStringType implements DBType<Boolean> {

	private final String sTrue;

	private final String sFalse;

	public BooleanStringType() {

		this("T", "F");
	}

	public BooleanStringType(final String sTrue, final String sFalse) {

		this.sTrue = sTrue;

		this.sFalse = sFalse;

	}

	@Override
	public boolean equals(final Object obj) {

		if (obj instanceof BooleanStringType) {

			final BooleanStringType bt = (BooleanStringType) obj;

			if (bt.sTrue.equals(this.sTrue) && bt.sFalse.equals(this.sFalse)) {

				return true;
			}
		}

		return false;
	}

	@Override
	public String toString() {

		return this.getClass().getSimpleName();
	}

	protected boolean getBooleanValue(final String s) throws SQLException {

		if (s == null) {
			return false;
		}

		if (s.equals("T")) {
			return true;
		}

		if (s.equals("F")) {
			return false;
		}

		if (s.equals("0")) {
			return false;
		}

		if (s.equals("1")) {
			return true;
		}

		if (s.equals("true")) {
			return true;
		}

		if (s.equals("false")) {
			return false;
		}

		throw new SQLException("value is not a boolean: " + s);

	}

	private static Boolean getBoolValue(final boolean b) {

		if (b) {
			return Boolean.TRUE;
		}

		return Boolean.FALSE;
	}

	@Override
	public Boolean getFromResultSet(final ResultSet rset, final int index) throws SQLException {

		final String s = rset.getString(index);

		final boolean b = getBooleanValue(s);

		return getBoolValue(b);
	}

	@Override
	public Boolean getFromResultSet(final ResultSet rset, final String field) throws SQLException {

		final String s = rset.getString(field);

		final boolean b = getBooleanValue(s);

		return getBoolValue(b);
	}

	@Override
	public Class<? extends Object> getTypeClass() {

		return Boolean.class;
	}

	@Override
	public void bindToStmt(final PreparedStatement stmt, final int index, final Boolean value) throws SQLException {

		if (value == null) {

			stmt.setString(index, null);

		} else {

			final String s = value ? sTrue : sFalse;

			stmt.setString(index, s);
		}
	}
}