package me.ahoo.eventbus.spring.boot.autoconfigure;


import com.mydao.eventbus.spring.annotation.EnableEventBus;
import me.ahoo.eventbus.core.compensation.PublishCompensation;
import me.ahoo.eventbus.core.compensation.SubscribeCompensation;
import me.ahoo.eventbus.core.compensation.impl.PublishCompensationImpl;
import me.ahoo.eventbus.core.compensation.impl.SubscribeCompensationImpl;
import me.ahoo.eventbus.core.compensation.impl.config.PublishConfig;
import me.ahoo.eventbus.core.compensation.impl.config.SubscribeConfig;
import me.ahoo.eventbus.core.consistency.ConsistencyPublisher;
import me.ahoo.eventbus.core.consistency.ConsistencySubscriberFactory;
import me.ahoo.eventbus.core.consistency.impl.ConsistencyPublisherImpl;
import me.ahoo.eventbus.core.consistency.impl.ConsistencySubscriberFactoryImpl;
import me.ahoo.eventbus.core.publisher.EventDescriptorParser;
import me.ahoo.eventbus.core.publisher.EventNameGenerator;
import me.ahoo.eventbus.core.publisher.Publisher;
import me.ahoo.eventbus.core.publisher.impl.SimpleEventDescriptorParser;
import me.ahoo.eventbus.core.publisher.impl.SimpleEventNameGenerator;
import me.ahoo.eventbus.core.repository.PublishEventRepository;
import me.ahoo.eventbus.core.repository.SubscribeEventRepository;
import me.ahoo.eventbus.core.subscriber.SubscriberNameGenerator;
import me.ahoo.eventbus.core.subscriber.SubscriberRegistry;
import me.ahoo.eventbus.core.subscriber.SubscriberScanner;
import me.ahoo.eventbus.core.subscriber.impl.SimpleSubscriberNameGenerator;
import me.ahoo.eventbus.jdbc.JdbcPublishEventRepository;
import me.ahoo.eventbus.jdbc.JdbcSubscribeEventRepository;
import me.ahoo.eventbus.rabbit.RabbitPublisher;
import me.ahoo.eventbus.rabbit.RabbitSubscriberRegistry;
import org.springframework.amqp.rabbit.connection.ConnectionFactory;
import org.springframework.amqp.rabbit.listener.RabbitListenerEndpointRegistry;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.transaction.PlatformTransactionManager;

/**
 * @author : ahoo wang
 */
@Configuration
@EnableEventBus
@EnableConfigurationProperties({EventBusProperties.class})
public class EventBusAutoConfiguration {

    private final EventBusProperties eventBusProperties;

    public EventBusAutoConfiguration(EventBusProperties eventBusProperties) {
        this.eventBusProperties = eventBusProperties;
    }

    @Bean
    @ConditionalOnMissingBean
    public EventNameGenerator eventNameGenerator() {
        return new SimpleEventNameGenerator();
    }

    @Bean
    @ConditionalOnMissingBean
    public SubscriberNameGenerator subscriberNameGenerator() {
        return new SimpleSubscriberNameGenerator(eventBusProperties.getSubscriber().getPrefix());
    }

    @Bean
    @ConditionalOnMissingBean
    public EventDescriptorParser eventDescriptorParser(EventNameGenerator eventNameGenerator) {
        return new SimpleEventDescriptorParser(eventNameGenerator);
    }

    @Bean
    @ConditionalOnMissingBean
    public SubscriberScanner subscriberScanner(SubscriberNameGenerator subscriberNameGenerator, EventDescriptorParser eventDescriptorParser) {
        return new SubscriberScanner(subscriberNameGenerator, eventDescriptorParser);
    }

    @Bean
    @ConditionalOnMissingBean
    public PublishEventRepository publishEventRepository(NamedParameterJdbcTemplate jdbcTemplate) {
        return new JdbcPublishEventRepository(jdbcTemplate);
    }

    @Bean
    @ConditionalOnMissingBean
    public SubscribeEventRepository subscribeEventRepository(NamedParameterJdbcTemplate jdbcTemplate) {
        return new JdbcSubscribeEventRepository(jdbcTemplate);
    }

    @Bean
    @ConditionalOnMissingBean
    public ConsistencyPublisher consistencyPublisher(Publisher publisher,
                                                     EventDescriptorParser eventDescriptorParser,
                                                     PublishEventRepository publishEventRepository,
                                                     PlatformTransactionManager transactionManager) {
        return new ConsistencyPublisherImpl(publisher, eventDescriptorParser, publishEventRepository, transactionManager);
    }

    @Bean
    @ConditionalOnMissingBean
    public ConsistencySubscriberFactory consistencySubscriberFactory(ConsistencyPublisher consistencyPublisher,
                                                                     EventDescriptorParser eventDescriptorParser,
                                                                     PublishEventRepository publishEventRepository,
                                                                     SubscribeEventRepository subscribeEventRepository,
                                                                     PlatformTransactionManager transactionManager) {
        return new ConsistencySubscriberFactoryImpl(consistencyPublisher, eventDescriptorParser, publishEventRepository, subscribeEventRepository, transactionManager);
    }


    @Configuration
    protected class RabbitConfiguration {

        @Bean
        @ConditionalOnMissingBean
        public Publisher publisher(
                ConnectionFactory connectionFactory) {
            return new RabbitPublisher(eventBusProperties.getRabbit(), connectionFactory);
        }

        @Bean
        public SubscriberRegistry subscribeContainer(
                ConnectionFactory connectionFactory,
                ConsistencySubscriberFactory subscriberFactory,
                RabbitListenerEndpointRegistry rabbitListenerEndpointRegistry) {
            return new RabbitSubscriberRegistry(eventBusProperties.getRabbit(), connectionFactory, subscriberFactory, rabbitListenerEndpointRegistry);
        }
    }

    @Configuration
    @ConditionalOnProperty(prefix = EventBusProperties.PREFIX, name = "compensation.enable", havingValue = "true", matchIfMissing = true)
    protected class CompensationConfiguration {
        @Bean
        @ConditionalOnMissingBean
        public PublishCompensation publishCompensation(
                ConsistencyPublisher consistencyPublisher,
                PublishEventRepository publishEventRepository) {
            PublishConfig publishConfig = EventBusAutoConfiguration.this.eventBusProperties.getCompensation().getPublish();
            return new PublishCompensationImpl(publishConfig, consistencyPublisher, publishEventRepository);
        }

        @Bean
        @ConditionalOnMissingBean
        public SubscribeCompensation subscribeCompensation(
                SubscriberRegistry subscriberRegistry,
                SubscribeEventRepository subscribeEventRepository) {
            SubscribeConfig subscribeConfig = EventBusAutoConfiguration.this.eventBusProperties.getCompensation().getSubscribe();
            return new SubscribeCompensationImpl(subscribeConfig, subscriberRegistry, subscribeEventRepository);
        }

    }
}
