/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package me.ahoo.cosec.spring.boot.starter.authentication

import me.ahoo.cosec.api.authentication.Authentication
import me.ahoo.cosec.api.authentication.AuthenticationProvider
import me.ahoo.cosec.api.authentication.Credentials
import me.ahoo.cosec.api.principal.CoSecPrincipal
import me.ahoo.cosec.authentication.CompositeAuthentication
import me.ahoo.cosec.authentication.DefaultAuthenticationProvider
import me.ahoo.cosec.spring.boot.starter.ConditionalOnCoSecEnabled
import org.springframework.boot.autoconfigure.AutoConfiguration
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean
import org.springframework.boot.context.properties.EnableConfigurationProperties
import org.springframework.context.ApplicationContext
import org.springframework.context.annotation.Bean
import org.springframework.context.annotation.Primary

/**
 * CoSec AutoConfiguration .
 *
 * @author ahoo wang
 */
@AutoConfiguration
@ConditionalOnCoSecEnabled
@ConditionalOnAuthenticationEnabled
@EnableConfigurationProperties(
    AuthenticationProperties::class,
)
class CoSecAuthenticationAutoConfiguration {
    @Bean
    @ConditionalOnMissingBean
    fun authenticationProvider(
        applicationContext: ApplicationContext
    ): AuthenticationProvider {
        applicationContext.getBeansOfType(Authentication::class.java).values.forEach {
            @Suppress("UNCHECKED_CAST")
            it as Authentication<Credentials, CoSecPrincipal>
            DefaultAuthenticationProvider.register(it)
        }
        return DefaultAuthenticationProvider
    }

    @Bean
    @Primary
    fun compositeAuthentication(authenticationProvider: AuthenticationProvider): CompositeAuthentication {
        return CompositeAuthentication(authenticationProvider)
    }
}
