package io.fincast.portfolio.impl

import io.fincast.enums.BookingKind
import io.fincast.enums.BookingType
import io.fincast.enums.ProductType
import io.fincast.portfolio.*

class PositionImpl(
	override val portfolio: Portfolio,
	override val tag: String,
	override val productType: ProductType,
) : Position {

	override var compos: List<PositionCompo> = emptyList()

	private val bookings: MutableList<Booking> = mutableListOf()

	private var reconDate: SimDate? = null

	private var balance = 0.0

	override fun handlePositionLifecycle(date: SimDate) {
		compos.forEach { it.handleLifecycle(date) }
	}

	override fun handlePortfolioLifecycle(date: SimDate) {
	}

	override fun isReconciled(date: SimDate): Boolean = reconDate != null && reconDate == date

	override fun getBalance(date: SimDate): Double {
		return getPeriod(date).amount
	}

	override fun getYield(date: SimDate): Double {
		return getPeriod(date).yield
	}

	override fun getTurnover(date: SimDate): Double {
		return getBalance(date) - getBalance(date - 1)
	}

	override fun getBookings(): List<Booking> {
		return bookings.filter { it.bookingKind != BookingKind.AGGREGATE }
	}

	override fun getBookings(date: SimDate): List<Booking> {
		return bookings.filter { it.date == date && it.bookingKind != BookingKind.AGGREGATE }
	}

	override fun getRefBookings(date: SimDate, refPos: Position): List<Booking> {
		return getBookings(date)
			.stream()
			.filter { b -> b is LifecycleBooking }
			.filter { b -> (b as LifecycleBooking).refPosition === refPos }
			.toList()
	}

	override fun getPeriods(): List<Booking> {
		return bookings.filter { it.bookingKind == BookingKind.AGGREGATE }
	}

	private fun getPeriod(date: SimDate): AggregateBooking {
		if (bookings.isEmpty() || date < getFirstDate()!!) {
			return AggregateBooking(this, date, 0.0, 0.0)
		} else if (date >= getLastDate()!!) {
			return AggregateBooking(this, date, balance, getYield())
		}
		return bookings.filter { it.date == date && it.bookingKind == BookingKind.AGGREGATE }[0] as AggregateBooking
	}

	override fun bookReconciliation(date: SimDate, balance: Double) {
		addBooking(ReconciliationBooking(this, date, balance))
	}

	override fun bookTurnover(date: SimDate, bookingKind: BookingKind, turnover: Double, refCompo: PositionCompo) {
		addBooking(LifecycleBooking(this, date, bookingKind, turnover, refCompo.position, refCompo))
	}

	override fun bookBalance(date: SimDate) {
		if (bookings.isNotEmpty()) {
			addBooking(AggregateBooking(this, date, balance, getYield()))
		}
	}

	private fun addBooking(booking: Booking) {
		require(booking.date >= (getLastDate() ?: booking.date))
		{ "bookings must be sequential i: ${booking.date} vs ${getLastDate()}" }
		require(booking.date <= (getLastDate() ?: booking.date) + 1)
		{ "bookings must be sequential ii: ${booking.date} vs ${getLastDate()}" }
		require(booking.bookingKind.bookingType == BookingType.AGGREGATE || reconDate == null || booking.date > reconDate!!)
		{ "cannot add booking to reconciled period" }
		if (booking.bookingKind.bookingType != BookingType.AGGREGATE && 0.0 == booking.amount) {
			return
		}
		when (booking.bookingKind.bookingType) {
			BookingType.RECONCILE -> {
				balance = booking.amount
				reconDate = booking.date
			}

			BookingType.TURNOVER -> {
				balance += booking.amount
			}

			BookingType.AGGREGATE -> {
				check(booking.amount == balance) { "balance must match" } // reporting only
			}
		}
		bookings.add(booking)
		portfolio.addBooking(booking)
	}

	private fun getFirstDate(): SimDate? {
		return if (bookings.isNotEmpty()) bookings[0].date else null
	}

	private fun getLastDate(): SimDate? {
		return if (bookings.isNotEmpty()) bookings[bookings.lastIndex].date else null
	}

}
