package io.fincast.portfolio

import io.fincast.enums.BookingKind

sealed interface Booking {
	val position: Position
	val date: SimDate
	val bookingKind: BookingKind
	val amount: Double
}

/**
 * A reconciliation half-booking on a position.
 *
 * A reconciliation booking sets the balance of a position at the given date.
 */
data class ReconciliationBooking(
	override val position: Position,
	override val date: SimDate,
	override val amount: Double,
) : Booking {

	override val bookingKind = BookingKind.RECONCILE

	override fun toString(): String {
		return "${date}: ${bookingKind.code} ${String.format("%.0f", amount)}"
	}

}

/**
 * A half-booking on a position, triggered by a position component lifecycle.
 */
data class LifecycleBooking(
	override val position: Position,
	override val date: SimDate,
	override val bookingKind: BookingKind,
	override val amount: Double,
	val refPosition: Position,
	val refCompo: PositionCompo,
) : Booking {

	override fun toString(): String {
		return "${date}: ${bookingKind.code} ${String.format("%.0f", amount)} ${refCompo.tag}"
	}

}

/**
 * An aggregate booking for a position.
 *
 * First level of aggregation, it reports the eop balance and yield of the position at a certain date.
 */
data class AggregateBooking(
	override val position: Position,
	override val date: SimDate,
	override val amount: Double,
	val yield: Double,
) : Booking {

	override val bookingKind = BookingKind.AGGREGATE

	override fun toString(): String {
		return "${date}: ${bookingKind.code} ${String.format("%.0f", amount)} ${String.format("%.0f", yield)}"
	}

}
