package io.fincast.household.impl

import io.fincast.portfolio.ValueProviders.constValue
import io.fincast.household.Valuable
import io.fincast.enums.ProductType
import io.fincast.enums.FundsAllocation
import io.fincast.portfolio.Portfolio
import io.fincast.portfolio.Position
import io.fincast.portfolio.PositionCompo
import io.fincast.portfolio.ValueProviders.eomCreditBalance
import io.fincast.portfolio.ValueProviders.eomDebitBalance
import io.fincast.enums.BookingKind
import io.fincast.enums.Periodicity
import io.fincast.portfolio.impl.YieldCompo

data class MoneyAccount(
	override val tag: String,
	override val reconBalance: Double = 0.0,
	val creditInterestRate: Double? = 0.0,
	val debitInterestRate: Double? = 0.0,
	val interestPeriodicity: Periodicity = Periodicity.YEARLY,
) : Valuable {

	override val productType = ProductType.VALUABLE

	class Builder {
		private var tag: String? = null
		private var reconBalance: Double? = null
		private var creditInterestRate: Double? = null
		private var debitInterestRate: Double? = null
		private var interestPeriodicity: Periodicity? = null
		fun tag(tag: String) = apply { this.tag = tag }
		fun reconBalance(reconBalance: Double) = apply { this.reconBalance = reconBalance }
		fun creditInterestRate(creditInterestRate: Double) = apply { this.creditInterestRate = creditInterestRate }
		fun debitInterestRate(debitInterestRate: Double) = apply { this.debitInterestRate = debitInterestRate }
		fun interestPeriodicity(interestPeriodicity: Periodicity) = apply { this.interestPeriodicity = interestPeriodicity }
		fun build(): MoneyAccount {
			return MoneyAccount(
				tag = tag ?: throw IllegalArgumentException("tag is required"),
				reconBalance = reconBalance ?: 0.0,
				creditInterestRate = creditInterestRate ?: 0.0,
				debitInterestRate = debitInterestRate ?: 0.0,
				interestPeriodicity = interestPeriodicity ?: Periodicity.YEARLY,
			)
		}
	}

	override fun createCompos(portfolio: Portfolio, pos: Position): List<PositionCompo> {
		val compos = mutableListOf<PositionCompo>()
		if (creditInterestRate != null && creditInterestRate != 0.0) {
			compos.add(
				YieldCompo(
					position = pos,
					tag = "creditInterest",
					amount = eomCreditBalance(pos),
					yieldRate = constValue(creditInterestRate),
					yieldPeriodicity = interestPeriodicity,
					bookingKind = BookingKind.INTEREST,
					fundsAllocation = FundsAllocation.COMPOUND,
				)
			)
		}
		if (debitInterestRate != null && debitInterestRate != 0.0) {
			compos.add(
				YieldCompo(
					position = pos,
					tag = "debitInterest",
					amount = eomDebitBalance(pos),
					yieldRate = constValue(debitInterestRate),
					yieldPeriodicity = interestPeriodicity,
					bookingKind = BookingKind.INTEREST,
					fundsAllocation = FundsAllocation.COMPOUND,
				)
			)
		}
		return compos
	}

}
