package io.fincast.household.impl

import io.fincast.portfolio.ValueProviders.constValue
import io.fincast.portfolio.ValueProviders.eomBalance
import io.fincast.household.Valuable
import io.fincast.enums.ProductType
import io.fincast.enums.FundsAllocation
import io.fincast.portfolio.Portfolio
import io.fincast.portfolio.Position
import io.fincast.portfolio.PositionCompo
import io.fincast.portfolio.SimDate
import io.fincast.enums.BookingKind
import io.fincast.enums.Periodicity
import io.fincast.portfolio.impl.CashflowCompo
import io.fincast.portfolio.impl.TransferCompo
import io.fincast.portfolio.impl.YieldCompo

data class InvestmentAccount(
	override val tag: String,
	override val reconBalance: Double = 0.0,

	val contributionAmount: Double? = null,
	val contributionPeriodicity: Periodicity? = null,
	val contributionStartDate: SimDate? = null,
	val contributionEndDate: SimDate? = null,

	val cashPart: Double = 0.0,
	val bondPart: Double = 0.0,
	val stockPart: Double = 0.0,
	val cashInterestRate: Double? = null,
	val bondInterestRate: Double? = null,
	val interestAllocation: FundsAllocation = FundsAllocation.COMPOUND,
	val stockDividendYield: Double? = null,
	val dividendAllocation: FundsAllocation = FundsAllocation.DISBURSE,
	val stockCapitalGain: Double? = null,

	val managementFee: Double? = 0.0,
	val performanceFee: Double? = 0.0,

	val withdrawalAmount: Double? = null,
	val withdrawalPeriodicity: Periodicity? = null,
	val withdrawalStartDate: SimDate? = null,
	val withdrawalEndDate: SimDate? = null,
) : Valuable {

	override val productType = ProductType.ASSET

	class Builder {
		private var tag: String? = null
		private var reconBalance: Double? = null
		private var contributionAmount: Double? = null
		private var contributionPeriodicity: Periodicity? = null
		private var contributionStartDate: SimDate? = null
		private var contributionEndDate: SimDate? = null
		private var cashPart: Double? = null
		private var bondPart: Double? = null
		private var stockPart: Double? = null
		private var cashInterestRate: Double? = null
		private var bondInterestRate: Double? = null
		private var interestAllocation: FundsAllocation? = null
		private var stockDividendYield: Double? = null
		private var dividendAllocation: FundsAllocation? = null
		private var stockCapitalGain: Double? = null
		private var managementFee: Double? = null
		private var performanceFee: Double? = null
		private var withdrawalAmount: Double? = null
		private var withdrawalPeriodicity: Periodicity? = null
		private var withdrawalStartDate: SimDate? = null
		private var withdrawalEndDate: SimDate? = null
		fun tag(tag: String) = apply { this.tag = tag }
		fun reconBalance(reconBalance: Double) = apply { this.reconBalance = reconBalance }
		fun contributionAmount(contributionAmount: Double) = apply { this.contributionAmount = contributionAmount }
		fun contributionPeriodicity(contributionPeriodicity: Periodicity) = apply { this.contributionPeriodicity = contributionPeriodicity }
		fun contributionStartDate(contributionStartDate: SimDate) = apply { this.contributionStartDate = contributionStartDate }
		fun contributionEndDate(contributionEndDate: SimDate) = apply { this.contributionEndDate = contributionEndDate }
		fun cashPart(cashPart: Double) = apply { this.cashPart = cashPart }
		fun bondPart(bondPart: Double) = apply { this.bondPart = bondPart }
		fun stockPart(stockPart: Double) = apply { this.stockPart = stockPart }
		fun cashInterestRate(cashInterestRate: Double) = apply { this.cashInterestRate = cashInterestRate }
		fun bondInterestRate(bondInterestRate: Double) = apply { this.bondInterestRate = bondInterestRate }
		fun interestAllocation(interestAllocation: FundsAllocation) = apply { this.interestAllocation = interestAllocation }
		fun stockDividendYield(stockDividendYield: Double) = apply { this.stockDividendYield = stockDividendYield }
		fun dividendAllocation(dividendAllocation: FundsAllocation) = apply { this.dividendAllocation = dividendAllocation }
		fun stockCapitalGain(stockCapitalGain: Double) = apply { this.stockCapitalGain = stockCapitalGain }
		fun managementFee(managementFee: Double) = apply { this.managementFee = managementFee }
		fun performanceFee(performanceFee: Double) = apply { this.performanceFee = performanceFee }
		fun withdrawalAmount(withdrawalAmount: Double) = apply { this.withdrawalAmount = withdrawalAmount }
		fun withdrawalPeriodicity(withdrawalPeriodicity: Periodicity) = apply { this.withdrawalPeriodicity = withdrawalPeriodicity }
		fun withdrawalStartDate(withdrawalStartDate: SimDate) = apply { this.withdrawalStartDate = withdrawalStartDate }
		fun withdrawalEndDate(withdrawalEndDate: SimDate) = apply { this.withdrawalEndDate = withdrawalEndDate }
		fun build(): InvestmentAccount {
			return InvestmentAccount(
				tag = tag ?: throw IllegalArgumentException("tag is required"),
				reconBalance = reconBalance ?: 0.0,
				contributionAmount = contributionAmount,
				contributionPeriodicity = contributionPeriodicity,
				contributionStartDate = contributionStartDate,
				contributionEndDate = contributionEndDate,
				cashPart = cashPart ?: 0.0,
				bondPart = bondPart ?: 0.0,
				stockPart = stockPart ?: 0.0,
				cashInterestRate = cashInterestRate,
				bondInterestRate = bondInterestRate,
				interestAllocation = interestAllocation ?: FundsAllocation.COMPOUND,
				stockDividendYield = stockDividendYield,
				dividendAllocation = dividendAllocation ?: FundsAllocation.DISBURSE,
				stockCapitalGain = stockCapitalGain,
				managementFee = managementFee ?: 0.0,
				performanceFee = performanceFee ?: 0.0,
				withdrawalAmount = withdrawalAmount,
				withdrawalPeriodicity = withdrawalPeriodicity,
				withdrawalStartDate = withdrawalStartDate,
				withdrawalEndDate = withdrawalEndDate,
			)
		}
	}

	override fun createCompos(portfolio: Portfolio, pos: Position): List<PositionCompo> {
		val compos: MutableList<PositionCompo> = mutableListOf()
		if (contributionAmount != null && contributionAmount > 0.0) {
			val startDate = SimDate.max(contributionStartDate ?: portfolio.reconDate, portfolio.reconDate)
			compos.add(
				TransferCompo(
					position = pos,
					tag = "contribution",
					fromPosition = portfolio.pocketMoney,
					toPosition = pos,
					amount = constValue(contributionAmount),
					sign = -1,
					startDate = startDate,
					endDate = contributionEndDate,
					periodicity = contributionPeriodicity ?: Periodicity.YEARLY,
				)
			)
		}
		if (cashPart != 0.0) {
			if (cashInterestRate != null && cashInterestRate != 0.0) {
				compos.add(
					YieldCompo(
						position = pos,
						tag = "cashInterest",
						amount = eomBalance(pos),
						yieldRate = constValue(cashPart / 100.0 * cashInterestRate),
						yieldPeriodicity = Periodicity.YEARLY,
						bookingKind = BookingKind.INTEREST,
						fundsAllocation = interestAllocation,
					)
				)
			}
		}
		if (bondPart != 0.0) {
			if (bondInterestRate != null && bondInterestRate != 0.0) {
				compos.add(
					YieldCompo(
						position = pos,
						tag = "bondInterest",
						amount = eomBalance(pos),
						yieldRate = constValue(bondPart / 100.0 * bondInterestRate),
						yieldPeriodicity = Periodicity.YEARLY,
						bookingKind = BookingKind.INTEREST,
						fundsAllocation = interestAllocation,
					)
				)
			}
		}
		if (stockPart != 0.0) {
			if (stockDividendYield != null && stockDividendYield != 0.0) {
				compos.add(
					YieldCompo(
						position = pos,
						tag = "dividend",
						amount = eomBalance(pos),
						yieldRate = constValue(stockPart / 100.0 * stockDividendYield),
						yieldPeriodicity = Periodicity.YEARLY,
						bookingKind = BookingKind.DIVIDEND,
						fundsAllocation = dividendAllocation,
					)
				)
			}
			if (stockCapitalGain != null && stockCapitalGain != 0.0) {
				compos.add(
					YieldCompo(
						position = pos,
						tag = "capitalGain",
						amount = eomBalance(pos),
						yieldRate = constValue(stockPart / 100.0 * stockCapitalGain),
						yieldPeriodicity = Periodicity.YEARLY,
						bookingKind = BookingKind.CAPITAL_GAIN,
						fundsAllocation = FundsAllocation.COMPOUND,
					)
				)
			}
		}
		if (managementFee != null && managementFee != 0.0) {
			compos.add(
				CashflowCompo(
					position = pos,
					tag = "managementFee",
					fundsAllocation = FundsAllocation.COMPOUND,
					amount = { managementFee / 100 * pos.getBalance(it) },
					sign = -1,
					periodicity = Periodicity.YEARLY,
				)
			)
		}
		if (performanceFee != null && performanceFee != 0.0) {
			compos.add(
				CashflowCompo(
					position = pos,
					tag = "performanceFee",
					fundsAllocation = FundsAllocation.COMPOUND,
					amount = { performanceFee / 100 * (pos.getBalance(it) - pos.getBalance(it.addYears(-1))) },
					sign = -1,
					periodicity = Periodicity.YEARLY,
				)
			)
		}
		if (withdrawalAmount != null && withdrawalAmount > 0.0) {
			val startDate = SimDate.max(withdrawalStartDate ?: portfolio.reconDate, portfolio.reconDate)
			compos.add(
				TransferCompo(
					position = pos,
					tag = "withdrawal",
					fromPosition = pos,
					toPosition = portfolio.pocketMoney,
					amount = constValue(withdrawalAmount),
					sign = -1,
					startDate = startDate,
					endDate = withdrawalEndDate,
					periodicity = withdrawalPeriodicity ?: Periodicity.YEARLY,
				)
			)
		}
		return compos
	}

}
