package io.fincast.household

import io.fincast.enums.BookingKind
import io.fincast.portfolio.SimDate

sealed interface Booking {
	val holding: Holding
	val date: SimDate
	val bookingKind: BookingKind
	val amount: Double
}

/**
 * A reconciliation half-booking on a holding.
 *
 * A reconciliation booking sets the balance of a holding at the given date.
 */
data class ReconciliationBooking(
	override val holding: Holding,
	override val date: SimDate,
	override val amount: Double,
) : Booking {

	override val bookingKind = BookingKind.RECONCILE

	override fun toString(): String {
		return "${date}: ${bookingKind.code} ${holding.tag} ${String.format("%.0f", amount)}"
	}

}

/**
 * A half-booking on a holding, triggered by a position component lifecycle.
 */
data class LifecycleBooking(
	override val holding: Holding,
	override val date: SimDate,
	override val bookingKind: BookingKind,
	override val amount: Double,
	val refHolding: Holding,
	val refCompo: String,
) : Booking {

	override fun toString(): String {
		return "${date}: ${bookingKind.code} ${holding.tag} ${String.format("%.0f", amount)} (${refHolding.tag}.${refCompo})"
	}

}

/**
 * An aggregate booking for a holding.
 *
 * An aggregate booking on a holding contains ending balance and yield (i.e. the aggregate of all bookings) for the period.
 */
data class AggregateBooking(
	override val holding: Holding,
	override val date: SimDate,
	override val amount: Double,
	val yield: Double,
) : Booking {

	override val bookingKind = BookingKind.AGGREGATE

	override fun toString(): String {
		return "${date}: ${bookingKind.code} ${holding.tag} ${String.format("%.0f", amount)} ${String.format("%.0f", yield)}"
	}

}
