package io.fincast.household

import io.fincast.engine.SimDate
import io.fincast.enums.Gender
import io.fincast.enums.Religion
import io.fincast.household.impl.PersonImpl

/**
 * A person is a member of a household, a partner or a child.
 *
 * @property tag tag to map back to client entity
 * @property birthDate the birthdate of the person
 * @property registeredGender the officially registered gender of this person (relevant for pension calculations)
 * @property religiousAffiliation the religious affiliation of the person
 * @property retirementAge the (potentially early) retirement age of the person, if omitted will be calculated as the standard retirement age
 * @property deathDate the (optional) death date of the person, if omitted the person will live until the end of the simulation
 */
interface Person {

	val tag: String

	val birthDate: SimDate

	val registeredGender: Gender

	val religiousAffiliation: Religion

	val retirementAge: Int?

	val deathDate: SimDate?

	/**
	 * Age in years of the person at the given date.
	 * @param date the date to calculate the age at
	 * @return the age in years of the person at the given date
	 */
	fun getAgeAt(date: SimDate): Int {
		return (date - birthDate) / SimDate.MONTHS_PER_YEAR
	}

	/**
	 * Actual retirement date of the person.
	 * If the retirement age is not given, retirement date is calculated based on the standard retirement age.
	 *
	 * @return the actual retirement date of the person
	 */
	fun getActualRetirementDate(): SimDate {
		return birthDate + getActualRetirementAge() * SimDate.MONTHS_PER_YEAR
	}

	/**
	 * Actual retirement age of the person.
	 * If the (early) retirement age is not given, the standard retirement age is returned.
	 *
	 * @return the actual retirement age of the person
	 */
	fun getActualRetirementAge(): Int {
		return retirementAge ?: getStatutoryRetirementAge()
	}

	/**
	 * Statutory retirement age in the country of the person.
	 * @return the statutory retirement age in the country of the person
	 */
	fun getStatutoryRetirementAge(): Int {
		return if (registeredGender == Gender.FEMALE) 64 else 65
	}

	/**
	 * End date of salary
	 * when no endDate is given, the minimum of the owners retirementDate and deathDate.
	 * when an endDate is given, the minimum of the endDate and the owners deathDate.
	 * retirementDate is overruled, since salary may continue beyond retirement.
	 */
	fun getEndOfSalaryDate(endDate: SimDate?): SimDate? {
		val deathDate = this.deathDate
		val retirementDate = this.getActualRetirementDate()
		if (endDate == null) {
			if (deathDate != null) return SimDate.min(retirementDate, deathDate)
			return retirementDate
		}
		return if (deathDate != null) SimDate.min(endDate, deathDate) else endDate
	}

	/**
	 * End date of expenses, the minimum of the endDate and the owners deathDate.
	 */
	fun getEndOfExpenseDate(endDate: SimDate?): SimDate? {
		val deathDate = this.deathDate
		return if (endDate == null) deathDate else if (deathDate != null) SimDate.min(endDate, deathDate) else endDate
	}

	class Builder {
		private var tag: String? = null
		private var birthDate: SimDate? = null
		private var registeredGender: Gender? = null
		private var religiousAffiliation: Religion? = null
		private var retirementAge: Int? = null
		private var deathDate: SimDate? = null
		fun tag(tag: String) = apply { this.tag = tag }
		fun birthDate(birthDate: SimDate) = apply { this.birthDate = birthDate }
		fun registeredGender(registeredGender: Gender) = apply { this.registeredGender = registeredGender }
		fun religiousAffiliation(religiousAffiliation: Religion) = apply { this.religiousAffiliation = religiousAffiliation }
		fun retirementAge(retirementAge: Int?) = apply { this.retirementAge = retirementAge }
		fun deathDate(deathDate: SimDate?) = apply { this.deathDate = deathDate }
		fun build(): Person {
			return PersonImpl(
				tag = tag ?: throw IllegalArgumentException("tag is required"),
				birthDate = birthDate ?: throw IllegalArgumentException("birthDate is required"),
				registeredGender = registeredGender ?: throw IllegalArgumentException("registeredGender is required"),
				religiousAffiliation = religiousAffiliation ?: Religion.OTHER,
				retirementAge = retirementAge,
				deathDate = deathDate,
			)
		}
	}

}
