package io.fincast.holding.impl.core

import io.fincast.compo.FinancialCompo
import io.fincast.compo.ValueProvider
import io.fincast.compo.impl.TransferCompo
import io.fincast.engine.SimDate
import io.fincast.enums.Periodicity
import io.fincast.enums.ProductType
import io.fincast.holding.Contract
import io.fincast.holding.Holding
import io.fincast.holding.ProjectionPhase
import io.fincast.holding.base.HoldingBase
import io.fincast.household.Person

/**
 * Value transfer between two Holdings
 *
 * Generates a (potentially periodic) transfer, between startDate and endDate.
 *
 * @property tag tag (non-unique) to map back to client entity
 * @property owner the (optional) owner of the holding
 * @property amount the amount to transfer at a given date
 * @property periodicity the periodicity of the transfer
 * @property startDate the (optional) start date of the transfer periodicity, if omitted will start at month following reconDate
 * @property endDate the (optional) end date of the transfer periodicity, or date of single transfer if periodicity is maturity
 */
data class Transfer(
	override val tag: String,
	override val owner: Person? = null,
	override val taxCode: String? = null,
	var fromHolding: Holding?,
	var toHolding: Holding?,
	val amount: ValueProvider<Double>,
	val periodicity: Periodicity = Periodicity.YEARLY,
	val projectionPhase: ProjectionPhase = ProjectionPhase.EOP_TRANSFER,
	override val startDate: SimDate? = null,
	override val endDate: SimDate? = null,
) : Contract, HoldingBase(tag, owner, ProductType.CONTRACT, taxCode) {

	class Builder {
		private var tag: String? = null
		private var taxCode: String? = null
		private var owner: Person? = null
		private var fromHolding: Holding? = null
		private var toHolding: Holding? = null
		private var amount: ValueProvider<Double>? = null
		private var periodicity: Periodicity? = null
		private var startDate: SimDate? = null
		private var endDate: SimDate? = null
		private var projectionPhase: ProjectionPhase? = null
		fun tag(tag: String) = apply { this.tag = tag }
		fun taxCode(taxCode: String?) = apply { this.taxCode = taxCode }
		fun owner(owner: Person?) = apply { this.owner = owner }
		fun fromHolding(fromHolding: Holding) = apply { this.fromHolding = fromHolding }
		fun toHolding(toHolding: Holding) = apply { this.toHolding = toHolding }
		fun amount(amount: ValueProvider<Double>) = apply { this.amount = amount }
		fun periodicity(periodicity: Periodicity?) = apply { this.periodicity = periodicity }
		fun startDate(startDate: SimDate?) = apply { this.startDate = startDate }
		fun endDate(endDate: SimDate?) = apply { this.endDate = endDate }
		fun projectionPhase(projectionPhase: ProjectionPhase?) = apply { this.projectionPhase = projectionPhase }
		fun build(): Transfer {
			return Transfer(
				tag = tag ?: throw IllegalArgumentException("tag is required"),
				taxCode = taxCode,
				owner = owner,
				fromHolding = fromHolding,
				toHolding = toHolding,
				amount = amount ?: throw IllegalArgumentException("amount is required"),
				periodicity = periodicity ?: Periodicity.YEARLY,
				startDate = startDate,
				endDate = endDate,
				projectionPhase = projectionPhase ?: ProjectionPhase.EOP_TRANSFER,
			)
		}
	}

	override fun createCompos(): List<FinancialCompo> {
		require(fromHolding != null) { "fromHolding is required" }
		require(toHolding != null) { "toHolding is required" }
		val endDate = owner?.getEndOfExpenseDate(endDate) ?: endDate
		val cf = TransferCompo(
			holding = this,
			tag = "transfer",
			fromHolding = fromHolding!!,
			toHolding = toHolding!!,
			amount = amount,
			periodicity = periodicity,
			startDate = startDate,
			endDate = endDate,
			projectionPhase = projectionPhase,
		)
		return listOf(cf)
	}

}
